<?php
/*
 * Benno MailArchiv
 *
 * Copyright  2008-2012 LWsystems GmbH & Co. KG
 *
 * http://www.lw-systems.de/
 * http://www.benno-mailarchiv.de/
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation.
 *
 * Binary versions of this file provided by LWsystems to you as
 * well as other copyrighted, protected or trademarked materials like
 * logos, graphics, fonts, specific documentations and configurations,
 * cryptographic keys etc. are subject to a license agreement between
 * you and LWsystems.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of the GNU Affero General Public License at
 * this URI:
 * http://www.gnu.org/licenses/agpl-3.0.html
 * If not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA  02111-1307  USA
 */

require_once 'model/DateConvert.php';

class Query
{

    /**
     * id
     */
    var $id;
    var $text;              // content or search term
    var $sender;
    var $recipient;
    var $subject;
    var $headerName;        // will only be set on extended search
    var $headerValue;       // will only be set on extended search
    var $isAttachment;
    var $isSpam;
    var $spamQuery;
    var $dateFrom;          // 01.01.1970
    var $dateTo;            // 31.12.2011
    var $type;              // Type of this query: Simple, Ext, Konv
    var $sort;              // DateAsc, SubjecDesc


    /**
     * Query constructor
     */
    function __construct($id,$text='',$dateFrom='',$dateTo='',$sender='',$recipient='',$subject='',$headerName='',$headerValue='',$isAttachment=false,$isSpam=false,$type='Simple')
    {
        global $App;

        $this->id = $id;
        $this->text = $text;          // content or search term
        $this->dateFrom = $dateFrom;
        $this->dateTo = $dateTo;
        $this->sender = trim($sender);
        $this->recipient = trim($recipient);
        $this->subject = $subject;
        $this->headerName = $headerName;        // only in extended search
        $this->headerValue = $headerValue;
        $this->isAttachment = $isAttachment;
        $this->isSpam = $isSpam;
        $this->sort  = 'DateAsc';
        $this->type = $type;

        $this->spamQuery = $App->getConfig('SPAM_QUERY','NOT HEADER-X-SPAM-FLAG:YES');
    }


    /**
     * returns formatted date
     *
     * @return string
     */
    function getDateFrom()
    {
        global $App;
        if ($this->dateFrom) {
            return strftime($App->date_format,$this->dateFrom);
        }
    }


    /**
     * returns formatted date
     *
     * @return string
     */
    function getDateTo()
    {
        global $App;
        if ($this->dateTo) {
            return strftime($App->date_format,$this->dateTo);
        }
    }


    /**
     * returns an array with query data
     *
     * @return array $query
     *
     */
    function getQuery ()
    {
        global $App;
        $query_string = '';

        if ($this->type == 'Simple') {            // Simple, Ext, Konv
            $query_string = $this->_simpleQuery();
        }
        else if ($this->type == 'RepoPath') {
            if ($this->text) {
                $idString = str_replace(':','/',$this->text);
                $query_string = '#RepoPath:'.$idString;
            }
            else {
                $query_string = 'id:notinindex';
            }
        }
        else {
            // extended query
            $query_string = $this->_extendedQuery();
        }

        // replace special chars
        $search = array('&');
        $replace = array('%26');
        // Achtung! Hier liegt ein Lucene Query-String vor!
        $query_string = str_replace($search,$replace,$query_string);

        return $query_string;
    }



    /**
     * returns url-safe string of serialized object
     *
     * @return string
     *
     */
    function getUrlParameter()
    {
        $QueryObject = $this;
        $jobject = json_encode($this);
        $queryData = base64_encode(gzcompress($jobject));

        return str_replace(array('+','/','='),array('-','_',''),$queryData);  // url safe encoding
    }


    /**
     * returns url-safe string of serialized object
     *
     * @return string
     *
     */
    static function getUrlObject($queryData)
    {

        $queryData64 = str_replace(array('-','_'),array('+','/'),$queryData);  // url safe decoding
        $queryDataGz = base64_decode($queryData64);
        $queryDataJs = gzuncompress($queryDataGz); 

        $Jobject = json_decode($queryDataJs);

        $QueryObject = new Query($Jobject->id);

        $QueryObject->id = $Jobject->id;
        $QueryObject->text = $Jobject->text;
        $QueryObject->dateFrom = $Jobject->dateFrom;
        $QueryObject->dateTo = $Jobject->dateTo;
        $QueryObject->sender = $Jobject->sender;
        $QueryObject->recipient = $Jobject->recipient;
        $QueryObject->subject = $Jobject->subject;
        $QueryObject->headerName = $Jobject->headerName;
        $QueryObject->headerValue = $Jobject->headerValue;
        $QueryObject->isAttachment = $Jobject->isAttachment;
        $QueryObject->isSpam = $Jobject->isSpam;
        $QueryObject->sort  = $Jobject->sort;
        $QueryObject->type = $Jobject->type;
        $QueryObject->spamQuery = $Jobject->spamQuery;
 
        // TEST:  echo -n '<urlparamer>'|tr - +|tr _ /|base64 -d|zlib-flate -uncompress
        return $QueryObject;
    }


    /**
     * returns base64 encoded attribute value
     *
     * @return string
     *
     */
    function getB64($attribute)
    {
        return base64_encode($this->$attribute);
    }

 

    /**
     * returns base64 encoded string for the simple search form
     *
     * @return string
     *
     */
    function getSimple ()
    {
        global $App;

        $field_text = '';
        if (! $this->headerName) {            // every other param could be ''
            $field_text = $this->text;
        }
        return base64_encode($field_text);
    }

 

    /**
     * Returns sort field setting
     */
    function getSortField()
    {

        if ($this->sort) {
            $sortField = preg_replace('/(.+)(Asc|Desc)$/','$1',$this->sort);
        }
        else {
            $sortField = 'Date';
        }

        return ucfirst($sortField);
    }

    /**
     * Returns if sort is ascending (true) or not ascending (false)
     */
    function getSortAsc()
    {
        $sortOrder = 'false';
        if (preg_match('/Asc$/',$this->sort)) {
            $sortOrder = true;
        }

        return $sortOrder;
    }
 

    /**
     * Query for spam mail too.
     *
     */  
    function setSpam($value=false)
    {
        $this->isSpam = $value;
    }


    /**
     * SPAM filter query
     *
     */  
    function setSpamQuery($query)
    {
        $this->spamQuery = $query;
    }


    /**
     * Set the sort field and order
     *
     * Possible values:
     *  - dateDesc
     *  - dateAsc
     */  
    function setSortOrder($sort='dateDesc')
    {
        $this->sort = $sort;
    }


    /**
     * simple query with only one search field
     *
     * @return string $query
     */
    function _simpleQuery ()
    {
        $fieldlist = array('Sender', 'Recipient', 'Subject', 'Text');

        if ($this->text) {
            $qstring = $this->text;
        }
        else {
            throw new Exception('QUERY_EMPTY');
        }

        $query_string = '('.$this->_format_query($qstring,$fieldlist).')';

        if (! $this->isSpam) {
            $query_string .= ' '.$this->spamQuery;
        }

        return $query_string;

    }


    /**
     * query with AND(ed) fields
     *
     * @return string $query
     */
    function _extendedQuery ()
    {
        global $App;

        $and   = ' AND ';
        $or    = ' OR ';

        $query_string = '';


        if ($this->type == 'Konv') {
            if ($this->sender && $this->recipient) {
                $p1s = $this->_asm_sender_query($this->sender);
                $p1r = $this->_asm_recipient_query($this->recipient);
                $p2s = $this->_asm_sender_query($this->recipient);
                $p2r = $this->_asm_recipient_query($this->sender);

                $query_string .= '(('.$p1s.$and.$p1r.')'.$or.
                                  '('.$p2s.$and.$p2r.'))'.$and;
            }  // Communication with one recipient will bei ORed
            else if (!$this->sender && $this->recipient) {
                $p1s = $this->_asm_sender_query($this->recipient);
                $p1r = $this->_asm_recipient_query($this->recipient);
                $query_string .= '('.$p1s.$or.$p1r.')'.$and;
            }
            else if (!$this->recipient && $this->sender) {
                $p1s = $this->_asm_sender_query($this->sender);
                $p1r = $this->_asm_recipient_query($this->sender);
                $query_string .= '('.$p1s.$or.$p1r.')'.$and;
            }
        }
        else {
            if ($this->sender) {
                $query_string .= $this->_asm_sender_query($this->sender).$and;
            }
            if ($this->recipient) {
                $query_string .= $this->_asm_recipient_query($this->recipient).$and;
            }
        }

        if ($this->subject !== '') {
            $attribute = str_replace('%','%25',$this->subject);
            $subject_string = $this->_format_query($attribute,'Subject');
            $query_string .= '('.$subject_string.')';
            $query_string .= $and;
        }

        if ($this->headerValue !== '') {
            $query_string .= '('.strtoupper('HEADER-'.$this->headerName).
                             ':"'.$this->headerValue.'")';
            $query_string .= $and;
        }

        if ($this->text !== '')   {
            $attribute = str_replace('%','%25',$this->text);
            $text_string = $this->_format_query($attribute,'Text');
            $query_string .= '('.$text_string.')';
            $query_string .= $and;
        }

        // search for all (*) messages for date-only queries
        if ($query_string == '') {
            if (($this->dateFrom != '') || ($this->dateTo != '')) {
                $query_string = '(Subject:*)'.$and;
            }
            else {
                throw new Exception('EQUERY_EMPTY');
            }
        }

        if ($this->isAttachment) {
            $query_string .= 'hasAttachment:[1 TO 9999]'.$and;
        }

        if (!$this->isSpam) {
            $query_string .= $this->spamQuery.$and;
        }

        $date_range   = $this->_dateRange($this->dateFrom,$this->dateTo);
        $query_string .= $date_range;
        $App->log("Querystring: $query_string",4,'Query');
        return $query_string;
    }


    /**
     * assemble sender query string
     */
    function _asm_sender_query ($participant) {
        if ($participant == '') { return ''; }

        global $App;
        //$or    = ' OR ';  // necessary??

        $sender_query = '';
        $sender_q = $participant.'*';
        if (!$App->getConfig('SUPPRESS_LEADING_WILDCARD')) {
            $sender_q = '*'.$participant.'*';
        }
        $sender_string = $this->_format_query($sender_q,'Sender');
        // (!)Sender contains plain email adress without names
        $from_string   = $this->_format_query($sender_q,'From');
        $sender_query .= '('.$sender_string.$or.$from_string.')';

        return $sender_query;
    }


    /**
     * assemble recipient query string
     */
    function _asm_recipient_query ($participant) {
        if ($participant == '') { return ''; }

        global $App;
        //$or    = ' OR ';  // necessary??

        $recipient_query = '';
        $recipient_q = $participant.'*';
        if (!$App->getConfig('SUPPRESS_LEADING_WILDCARD')) {
            $recipient_q = '*'.$participant.'*';
        }
        // (!)Recipient contains plain email adress without names
        $recipient_string = $this->_format_query($recipient_q,'Recipient');
        $to_string        = $this->_format_query($recipient_q,'To');
        $cc_string        = $this->_format_query($recipient_q,'Cc');
        $recipient_query .= '('.$recipient_string.$or.
                             $to_string.$or.
                             $cc_string.')';

        return $recipient_query;
    }


    /**
     * format querystring
    */
    function _format_query($qstring,$field)
    {
        global $App;

        if (is_array($field)) {
            $fieldlist = $field;
        }
        else {
            $fieldlist = array($field);
        }
        $qstring=addcslashes($qstring,'!(){}[]^:\'');
        $tokenlist = $this->_tokenize_query($qstring);

        $query_string = '';
        foreach ($fieldlist as $field) {
            $field_string = $this->_format_fieldstring($field,$tokenlist);
            $query_string .= '('.$field_string.')';
        }

        if ($App->getConfig('SUPPRESS_SINGLE_WILDCARD')) {
            if (preg_grep('/^\*$/',$tokenlist)) {
                $App->log('[Query] Single wildcard in in query:'.$query_string,1);
                throw new Exception('LEADING_WILDCARD');
            }
        }

        if ($App->getConfig('SUPPRESS_LEADING_WILDCARD')) {
            if (preg_match('/:[\*\?]/',$query_string)) {
                $App->log('[Query] Remove leading wildcards from query.',1);
                throw new Exception('LEADING_WILDCARD');
            }
        }

        return $query_string;
    }

    /**
     * Tokenize querystring considering boolean prefixes (+-)
     */
    function _tokenize_query($qstring)
    {
        global $App;

        $tokenlist = array();
        while (strlen($qstring)) {
            $qstring = trim($qstring);
            $matches = array();

            if ($App->getConfig('AUTO_ENCLOSE_DASHES',true)) {
            // match the first 'word - word' without enclosing "..."
            if (preg_match('/^[^"](\S+\s+[-+]\s+\S+)/',$qstring,$matches)) {
                array_push($tokenlist,'"'.$matches[0].'"');
                $qstring = preg_replace('/^(\S+\s+[+-]\s+\S+)/','',$qstring);
                continue;
            }
            }

            // ^\S matches "\S, thus check literal in "..." first
            if (preg_match('/^([+-]?"[^"]+")/',$qstring,$matches)) {
                array_push($tokenlist,$matches[0]);
                $qstring = preg_replace('/^([+-]?"[^"]+")/','',$qstring);
                continue;
            }
            // match first "word" in string
            if (preg_match('/^(\S+\s*?)/',$qstring,$matches)) {
                array_push($tokenlist,$matches[0]);
                $qstring = preg_replace('/^(\S+)/','',$qstring);
                continue;
            }
        }

        // escape +- general escaping in -> formatquery():addcslashes
        $tokenlist = preg_replace('/^([-+])$/','\\\$1',$tokenlist);

        return $tokenlist;
    } 


    function _format_fieldstring($field_name,$tokenlist)
    {
            // prepare for field
            $field_string = '';
            foreach ($tokenlist as $token) {
                $prefix = '';
                if (preg_match('/^([\+-])/',$token,$match)) {
                    $prefix = $match[0];
                }
                $token = preg_replace('/^[\+-]/','',$token,-1);
                $encprefix = urlencode($prefix);
                $field_string .= $encprefix.$field_name.':'.$token.' ';
            }
            $field_string = preg_replace("/\s$/",'',$field_string);

            return $field_string;
    }


    /**
     * return the lucene string for the date range
     */
    function _dateRange ($first,$last)
    {
        if (!$first) {
            $first = 0;
        }
        if (!$last) {
            $last = 2147483647;
        }
        $firstString = strftime('%Y%m%d%H%M',$first);
        $lastString  = strftime('%Y%m%d%H%M',$last);
	    $lastString  = substr($lastString, 0, -4)."2400"; 
        return "Date:[$firstString%20TO%20$lastString]";
    }


    /**
     * compatiblity to old (< v2.10.3) stored queries -> skip archive
     */
    function __unserialize(array $data): void
    {
        $this->id           = $data['id'];
        $this->text         = $data['text'];
        $this->dateFrom     = $data['dateFrom'];
        $this->dateTo       = $data['dateTo'];
        $this->sender       = $data['sender'];
        $this->recipient    = $data['recipient'];
        $this->subject      = $data['subject'];
        $this->headerName   = $data['headerName'];
        $this->headerValue  = $data['headerValue'];
        $this->isAttachment = $data['isAttachment'];
        $this->isSpam       = $data['isSpam'];
        $this->sort         = $data['sort'];
        $this->type         = $data['type'];
        $this->spamQuery    = $data['spamQuery'];
    }

}

?>
