package Benno::LDAP;

use Net::LDAP;
use Carp;
use parent qw(Benno);


=head1 NAME

Benno::LDAP

=head1 SYNOPSIS

    use C<Benno::LDAP>;

=head1 DESCRIPTION

=head1 METHODS

=head2 new($config)

Create new Benno::LDAP object and bind to ldap directory.

Throws exception if failed.

=cut
sub new
{
    my $this = shift;
    my $class = ref($this) || $this;
    my ($config) = shift;

    my $self = {};
    bless $self, $class;

    $self->{DEBUG}       = $config->{DEBUG};
    $self->{host}        = $config->{host};
    $self->{basedn}      = $config->{basedn};
    $self->{binddn}      = $config->{binddn};
    $self->{password}    = $config->{password};
    $self->{userattr}    = $config->{userattr};
    $self->{objectclass} = $config->{objectclass};
    $self->{attributes}  = $config->{attributes};
    $self->{tls}         = $config->{tls};

    $self->_connect_ldap($config);

    return $self;
}


# connect_ldap
sub _connect_ldap
{
    my ($self) = @_;

    my @hosts = $self->_get_hostlist();

    my $LDAP;
    foreach my $ldapstr (@hosts) {
        next unless $ldapstr;
        $self->{DEBUG} && print STDERR "Try to connect $ldapstr\n";
        my ($ldaphost,$ldapport) = split /:/, $ldapstr;
        eval {
            if ($self->{tls} eq 'ldaps') {
                $ldapport = 636 unless $ldapport;
                $LDAP = Net::LDAP->new('ldaps://'.$ldaphost, port => $ldapport,
                                       verify => 'none') or die "$@\n";
            }
            else {
                $ldapport = 389 unless $ldapport;
                $LDAP = Net::LDAP->new($ldaphost,
                                       port => $ldapport) or die "$@\n";
            }
        };
        last if $LDAP;
    }
    if ($@) {
        die "ERROR connect LDAP host: $@";
    }

    $self->{DEBUG} && $LDAP->debug($self->{DEBUG});

    my $mesg;
    if ($self->{tls} eq 'true') {
        $mesg = $LDAP->start_tls(verify => 'none');
        $mesg->code and exit 1;
    }
    if ($self->{binddn}) {
        $mesg = $LDAP->bind($self->{binddn},'password' => $self->{password});
    }
    else {
        $mesg = $LDAP->bind();      # anonymous bind
    }
    $mesg->code and die $mesg->error;
    $self->{_LDAP} = $LDAP;
}


# _get_hostlist
sub _get_hostlist
{
    return split /,(\s+)?/,$_[0]->{host};
}


# get_user($userid)
#
# returns a Net::LDAP::Entry object of the user
sub get_user
{
    my $self = shift;
    my $userid = shift;
    my $userfilter = shift;

    my $filter;
    if ($userfilter) {
        $filter = $userfilter;
    }
    else {
        $filter = "(&(objectClass=$self->{objectclass})($self->{userattr}=%s))";
    }

    my ($localpart,$domainpart) = $userid =~ /^(.+)\@(.+)$/;
    $filter =~ s/%u/$localpart/g;
    $filter =~ s/%d/$domainpart/g;
    $filter =~ s/%s/$userid/g;
    print "LDAP Search: \"$filter\"\n" if $verbose;
    my $mesg = $self->{_LDAP}->search( base   => $self->{basedn},
                                       attrs  => $self->{attributes},
                                       filter => $filter
                                     );
    if ($mesg->code) {
        print STDERR "Error search \"$filter\": ",$mesg->error,".\n" if $self->{DEBUG};
    }

    if ($mesg->count < 1) {
        print STDERR "No match for \"$filter\" in directory.\n" if $self->{DEBUG};
    }

    if ($mesg->count > 1) {
        die "Userid not unique at ldap directory.\n" if $self->{DEBUG};
    }
    return $mesg->entry(0);
}

# user_auth($userdn,$userpass)
#
#
sub user_auth
{
    my $self = shift;
    my ($userdn,$userpass) = @_;
    my $mesg = $self->{_LDAP}->bind($userdn,'password' => $userpass);
    if ($mesg->code) {
        return 'ERROR AUTHENTICATION ERROR';
    }

    return 'OK';
}


# get_groups($userid,$config)
#
# returns a list of Net::LDAP::Entry group objects
sub get_groups
{
    my ($self,$userid,$config) = @_;

    my @grouplist;

    my $filter;
    if ($config->{groupfilter}) {
        $filter = $config->{groupfilter};
    }
    else {
        $filter = "(&(objectClass=$config->{groupobjectclass})($config->{groupuserattr}=$userid))";
    }

    print "LDAP Search: \"$filter\"\n" if $verbose;
    my $mesg = $self->{_LDAP}->search( base   => $self->{basedn},
                                       filter => $filter
                                     );
    if ($mesg->code) {
        print STDERR "Error search \"$filter\": ",$mesg->error,".\n";
    }

    if ($mesg->count < 1) {
        print STDERR "No match for \"$filter\" in directory.\n";
    }

    return $mesg->entries;
}

# get_user_dn($uid,$config)
#
# gets DN of user from uid and userid attribute
sub get_user_dn
{
	my ($self,$uid,$config) = @_;

	my $filter = "(&(objectclass=$config->{objectclass})($config->{userattr}=$uid))";

    print "LDAP Search: \"$filter\"\n" if $verbose;
	my $mesg = $self->{_LDAP}->search(	base	=> $self->{basedn},
										filter	=> $filter
									);
	if ($mesg->code) {
		print STDERR "Error search \"$filter\": ",$mesg->error,".\n";
	}

	if ($mesg->count < 1) {
		print STDERR "No match for \"$filter\" in directory.\n";
	}

	# how do I get the DN of the first entry?
	my @entries = $mesg->entries;
	my $first = $entries[0];
	my $dn = $first->dn();

	return $dn;
}

### EOP ###
1;
