/*
 * Decompiled with CFR 0.152.
 */
package de.lwsystems.mailarchive.bennoauth.common.utils;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.jwk.Curve;
import com.nimbusds.jose.jwk.ECKey;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jose.jwk.gen.ECKeyGenerator;
import com.nimbusds.jose.jwk.gen.RSAKeyGenerator;
import de.lwsystems.mailarchive.bennoauth.common.utils.PemUtils;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECParameterSpec;
import java.security.spec.InvalidKeySpecException;
import java.util.Base64;
import java.util.Optional;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

@Component
public class KeyGeneratorUtils {
    private static final Logger LOGGER = LogManager.getLogger(KeyGeneratorUtils.class);
    private static final String EC_KEY_GENERATOR_ALGORITHM = "EC";
    @Value(value="${benno.jwks.ec.publicKey:#{null}}")
    private String ellipticPublicKey;
    @Value(value="${benno.jwks.ec.privateKey:#{null}}")
    private String ellipticPrivateKey;
    @Value(value="${benno.jwks.jks.path:#{null}}")
    private String bennoJksPath;
    @Value(value="${benno.jwks.jks.alias:#{null}}")
    private String bennoJksAlias;
    @Value(value="${benno.jwks.jks.keystoreType:pkcs12}")
    private String bennoJksKeystoreType;
    @Value(value="${benno.jwks.jks.password:#{null}}")
    private String bennoJksPassword;
    private ECPrivateKey privateKey = null;
    private ECPublicKey publicKey = null;

    public RSAKey generateRsaKey() throws JOSEException {
        Optional<RSAKey> rsaKeyFromKeyStore = this.generateRsaKeyFromJks();
        if (rsaKeyFromKeyStore.isPresent()) {
            LOGGER.info("The configured JKS has been registered in the JWKS.");
            return rsaKeyFromKeyStore.get();
        }
        LOGGER.warn("The JKS configured by the system has been registered in the JWKS. It is valid until the Benno-Auth server is terminated.");
        return this.generateDefaultRsaKey();
    }

    public ECKey generateEcKey() throws JOSEException {
        Optional<ECKey> ecKeyWithKeyPair = this.generateEcKeyWithKeyPair();
        if (ecKeyWithKeyPair.isPresent()) {
            LOGGER.info("The configured JKS has been registered in the JWKS.");
            return ecKeyWithKeyPair.get();
        }
        LOGGER.warn("The EC configured by the system has been registered in the JWKS. It is valid until the Benno-Auth server is terminated.");
        return this.generateDefaultEcKey();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Optional<RSAKey> generateRsaKeyFromJks() {
        try (FileInputStream bennoJksFile = new FileInputStream(this.bennoJksPath);){
            KeyStore javaKeyStore;
            switch (this.bennoJksKeystoreType) {
                case "pkcs12": {
                    javaKeyStore = KeyStore.getInstance("pkcs12");
                    break;
                }
                case "jks": {
                    javaKeyStore = KeyStore.getInstance("jks");
                    break;
                }
                default: {
                    LOGGER.error("Given Keystore Type not exists.");
                    Optional<RSAKey> optional = Optional.empty();
                    return optional;
                }
            }
            if (this.bennoJksPassword == null) {
                LOGGER.error("No JKS password set");
                return Optional.empty();
            }
            char[] jksPassword = this.bennoJksPassword.toCharArray();
            javaKeyStore.load(bennoJksFile, jksPassword);
            Optional<RSAKey> optional = Optional.ofNullable(RSAKey.load((KeyStore)javaKeyStore, (String)this.bennoJksAlias, (char[])jksPassword));
            return optional;
        }
        catch (IOException ioException) {
            LOGGER.error("There is an IO Error with JKS.", (Throwable)ioException);
            return Optional.empty();
        }
        catch (KeyStoreException keyStoreException) {
            LOGGER.error("Can't open JKS.", (Throwable)keyStoreException);
            return Optional.empty();
        }
        catch (CertificateException certificateException) {
            LOGGER.error("Certificate of JKS can't be loaded.", (Throwable)certificateException);
            return Optional.empty();
        }
        catch (NoSuchAlgorithmException noSuchAlgorithmException) {
            LOGGER.error("Given algorithm not found.", (Throwable)noSuchAlgorithmException);
            return Optional.empty();
        }
        catch (JOSEException joseException) {
            LOGGER.error("Can't load RSAKey.", (Throwable)joseException);
            return Optional.empty();
        }
        catch (NullPointerException nullPointerException) {
            LOGGER.warn("No JKS found.");
        }
        return Optional.empty();
    }

    private RSAKey generateDefaultRsaKey() throws JOSEException {
        return new RSAKeyGenerator(2048).generate();
    }

    private Optional<ECKey> generateEcKeyWithKeyPair() {
        if (this.ellipticPublicKey != null) {
            Path publicKeyPath = Paths.get(this.ellipticPublicKey, new String[0]);
            try {
                if (Files.isRegularFile(publicKeyPath, new LinkOption[0])) {
                    this.generateKeysFromFiles(publicKeyPath);
                } else {
                    this.generateKeysFromStrings();
                }
                return this.buildEcKey();
            }
            catch (IOException ex) {
                LOGGER.error("There is an IO Error.", (Throwable)ex);
            }
            catch (NoSuchAlgorithmException ex) {
                LOGGER.error("Given algorithm not found.", (Throwable)ex);
            }
            catch (InvalidKeySpecException ex) {
                LOGGER.error("Couldn't reconstruct public key.", (Throwable)ex);
            }
            catch (InvalidPathException ex) {
                LOGGER.error("Given path of key files are not correct.", (Throwable)ex);
            }
            catch (SecurityException ex) {
                LOGGER.error("Can't read key file.", (Throwable)ex);
            }
            catch (NullPointerException ex) {
                LOGGER.error("Public/Private key are not valid.", (Throwable)ex);
            }
        } else {
            LOGGER.info("No EC Public key found for JWKSource.");
        }
        return Optional.empty();
    }

    private Optional<ECKey> buildEcKey() {
        Curve curve = Curve.forECParameterSpec((ECParameterSpec)this.publicKey.getParams());
        if (this.privateKey != null) {
            return Optional.ofNullable(new ECKey.Builder(curve, this.publicKey).privateKey(this.privateKey).build());
        }
        return Optional.ofNullable(new ECKey.Builder(curve, this.publicKey).build());
    }

    private void generateKeysFromFiles(Path publicKeyPath) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        this.generatePrivateKeyFromFile();
        this.generatePublicKeyFromFile(publicKeyPath);
    }

    private void generatePrivateKeyFromFile() throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        Path privateKeyPath;
        if (this.ellipticPrivateKey != null && Files.isRegularFile(privateKeyPath = Paths.get(this.ellipticPrivateKey, new String[0]), new LinkOption[0])) {
            this.privateKey = (ECPrivateKey)PemUtils.readPrivateKeyFromFile(privateKeyPath.toAbsolutePath().toString(), EC_KEY_GENERATOR_ALGORITHM);
        }
    }

    private void generatePublicKeyFromFile(Path publicKeyPath) throws IOException, InvalidKeySpecException, NoSuchAlgorithmException {
        this.publicKey = (ECPublicKey)PemUtils.readPublicKeyFromFile(publicKeyPath.toAbsolutePath().toString(), EC_KEY_GENERATOR_ALGORITHM);
    }

    private void generateKeysFromStrings() throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        this.generatePrivateKeyFromString();
        this.generatePublicKeyFromString();
    }

    private void generatePrivateKeyFromString() throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        if (this.ellipticPrivateKey != null) {
            this.privateKey = (ECPrivateKey)PemUtils.readPrivateKeyFromString(new String(Base64.getDecoder().decode(this.ellipticPrivateKey)), EC_KEY_GENERATOR_ALGORITHM);
        }
    }

    private void generatePublicKeyFromString() throws IOException, InvalidKeySpecException, NoSuchAlgorithmException {
        this.publicKey = (ECPublicKey)PemUtils.readPublicKeyFromString(new String(Base64.getDecoder().decode(this.ellipticPublicKey.getBytes(StandardCharsets.UTF_8))), EC_KEY_GENERATOR_ALGORITHM);
    }

    private ECKey generateDefaultEcKey() throws JOSEException {
        return new ECKeyGenerator(Curve.SECP256K1).generate();
    }
}

