/*
 * Decompiled with CFR 0.152.
 */
package de.lwsystems.mailarchive.journal;

import de.lwsystems.mailarchive.archive.journal.JournalReader;
import de.lwsystems.mailarchive.journal.ChecksumEntry;
import de.lwsystems.mailarchive.journal.ChecksumValidator;
import de.lwsystems.mailarchive.journal.JournalEntry;
import de.lwsystems.mailarchive.journal.JournalOperation;
import de.lwsystems.mailarchive.journal.JournalParser;
import de.lwsystems.mailarchive.utils.ReverseLineReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class DefaultJournalParser
implements JournalParser {
    private static final Logger LOGGER = LogManager.getLogger(DefaultJournalParser.class);
    private static final Pattern JOURNAL_LINE_PATTERN = Pattern.compile("^(\\d{4}-\\d{2}-\\d{2} \\d{2}:\\d{2}:\\d{2} UTC): \\((SUCCESS)\\) (ARCHIVED)\\s+\"([^\"]+)\"(?:\\s*:.*)?$", 2);
    private static final Pattern CHECKSUM_PATTERN = Pattern.compile("[0-9A-F]{66}");
    private static final Pattern YEAR_BOX_PATTERN = Pattern.compile("(\\d{4}):[0-9A-F]{66}");

    @Override
    public Iterator<JournalEntry> parseFile(Path journalFile, JournalReader reader, boolean reverse) throws IOException {
        if (reverse) {
            ReverseLineReader reverseReader = new ReverseLineReader(journalFile);
            return new ReverseJournalEntryIterator(reverseReader, journalFile);
        }
        BufferedReader bufferedReader = reader.createBufferedReaderFromPath(journalFile);
        return new JournalEntryIterator(bufferedReader, journalFile);
    }

    private Optional<JournalEntry> parseLine(String line) {
        Optional<JournalEntry> journalEntry = Optional.empty();
        if (line == null || line.trim().isEmpty()) {
            return journalEntry;
        }
        Matcher matcher = JOURNAL_LINE_PATTERN.matcher(line);
        if (matcher.matches()) {
            String timestamp = matcher.group(1);
            boolean success = "SUCCESS".equals(matcher.group(2));
            String status = matcher.group(3);
            String content = matcher.group(4);
            JournalOperation operation = JournalOperation.fromStatus(status);
            journalEntry = this.extractChecksumEntry(content).map(checksumEntry -> new JournalEntry(timestamp, success, operation, (ChecksumEntry)checksumEntry));
        }
        if (journalEntry.isEmpty()) {
            return this.tryAlternativeFormat(line);
        }
        return journalEntry;
    }

    private Optional<ChecksumEntry> extractChecksumEntry(String content) {
        String[] parts;
        if (content == null || content.trim().isEmpty()) {
            return Optional.empty();
        }
        if (content.contains(":") && (parts = content.split(":", 2)).length == 2 && this.isValidFullChecksum(parts[1])) {
            return Optional.of(new ChecksumEntry(content, parts[1], parts[0]));
        }
        return Optional.empty();
    }

    private Optional<JournalEntry> tryAlternativeFormat(String line) {
        String checksum;
        JournalOperation operation = JournalOperation.fromLine(line);
        if (operation != JournalOperation.ARCHIVED) {
            return Optional.empty();
        }
        Matcher checksumMatcher = YEAR_BOX_PATTERN.matcher(line);
        if (checksumMatcher.find() && this.isValidFullChecksum(checksum = checksumMatcher.group())) {
            String yearBox = this.extractYearBox(line, checksum).orElse("");
            ChecksumEntry entry = new ChecksumEntry(checksum, checksum, yearBox);
            if (!yearBox.isEmpty()) {
                return Optional.of(new JournalEntry("unknown", true, operation, entry));
            }
        }
        return Optional.empty();
    }

    private Optional<String> extractYearBox(String line, String checksum) {
        Matcher matcher = YEAR_BOX_PATTERN.matcher(line);
        while (matcher.find()) {
            if (!matcher.group().endsWith(checksum)) continue;
            return Optional.of(matcher.group(1));
        }
        return Optional.empty();
    }

    private boolean isValidFullChecksum(String checksum) {
        return ChecksumValidator.isValid66Checksum(checksum);
    }

    private class ReverseJournalEntryIterator
    implements Iterator<JournalEntry> {
        private final ReverseLineReader reverseReader;
        private final Path journalFile;
        private JournalEntry nextEntry;
        private int lineNumber;
        private int entriesFound;
        private boolean closed;

        ReverseJournalEntryIterator(ReverseLineReader reverseReader, Path journalFile) {
            this.reverseReader = reverseReader;
            this.journalFile = journalFile;
            this.lineNumber = reverseReader.getTotalLines() + 1;
            this.entriesFound = 0;
            this.closed = false;
            this.advanceToNext();
        }

        @Override
        public boolean hasNext() {
            return this.nextEntry != null;
        }

        @Override
        public JournalEntry next() {
            if (this.nextEntry == null) {
                throw new NoSuchElementException("No more journal entries available");
            }
            JournalEntry result = this.nextEntry;
            this.advanceToNext();
            return result;
        }

        private void advanceToNext() {
            this.nextEntry = null;
            if (this.closed) {
                return;
            }
            try {
                String line;
                while ((line = this.reverseReader.readLine()) != null) {
                    --this.lineNumber;
                    try {
                        Optional<JournalEntry> entry = DefaultJournalParser.this.parseLine(line);
                        if (!entry.isPresent()) continue;
                        this.nextEntry = entry.get();
                        ++this.entriesFound;
                        return;
                    }
                    catch (Exception e) {
                        LOGGER.debug("Error parsing line {} in file {}: {}", (Object)this.lineNumber, (Object)this.journalFile, (Object)e.getMessage());
                    }
                }
                this.closeQuietly();
                LOGGER.trace("Parsed {} entries from {} (reverse reading)", (Object)this.entriesFound, (Object)this.journalFile);
            }
            catch (IOException e) {
                LOGGER.warn("Error reading journal file {} in reverse: {}", (Object)this.journalFile, (Object)e.getMessage());
                this.closeQuietly();
            }
        }

        private void closeQuietly() {
            if (!this.closed) {
                try {
                    this.reverseReader.close();
                }
                catch (IOException e) {
                    LOGGER.debug("Error closing journal file {}: {}", (Object)this.journalFile, (Object)e.getMessage());
                }
                this.closed = true;
            }
        }
    }

    private class JournalEntryIterator
    implements Iterator<JournalEntry> {
        private final BufferedReader bufferedReader;
        private final Path journalFile;
        private JournalEntry nextEntry;
        private int lineNumber;
        private int entriesFound;
        private boolean closed;

        JournalEntryIterator(BufferedReader bufferedReader, Path journalFile) {
            this.bufferedReader = bufferedReader;
            this.journalFile = journalFile;
            this.lineNumber = 0;
            this.entriesFound = 0;
            this.closed = false;
            this.advanceToNext();
        }

        @Override
        public boolean hasNext() {
            return this.nextEntry != null;
        }

        @Override
        public JournalEntry next() {
            if (this.nextEntry == null) {
                throw new NoSuchElementException("No more journal entries available");
            }
            JournalEntry result = this.nextEntry;
            this.advanceToNext();
            return result;
        }

        private void advanceToNext() {
            this.nextEntry = null;
            if (this.closed) {
                return;
            }
            try {
                String line;
                while ((line = this.bufferedReader.readLine()) != null) {
                    ++this.lineNumber;
                    try {
                        Optional<JournalEntry> entry = DefaultJournalParser.this.parseLine(line);
                        if (!entry.isPresent()) continue;
                        this.nextEntry = entry.get();
                        ++this.entriesFound;
                        return;
                    }
                    catch (Exception e) {
                        LOGGER.debug("Error parsing line {} in file {}: {}", (Object)this.lineNumber, (Object)this.journalFile, (Object)e.getMessage());
                    }
                }
                this.closeQuietly();
                LOGGER.trace("Parsed {} entries from {}", (Object)this.entriesFound, (Object)this.journalFile);
            }
            catch (IOException e) {
                LOGGER.warn("Error reading journal file {}: {}", (Object)this.journalFile, (Object)e.getMessage());
                this.closeQuietly();
            }
        }

        private void closeQuietly() {
            if (!this.closed) {
                try {
                    this.bufferedReader.close();
                }
                catch (IOException e) {
                    LOGGER.debug("Error closing journal file {}: {}", (Object)this.journalFile, (Object)e.getMessage());
                }
                this.closed = true;
            }
        }
    }
}

