#
# -*- coding: utf-8 -*-
#
# Benno MailArchiv
#
# Copyright 2018-2021 LWsystems GmbH
#
# http://www.lw-systems.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.
#
import hashlib
import tempfile
import imaplib
import json
import os
import re


class Mailbox:
    """IMAP connection to mailbox"""

    def __init__(self, logger, imaphost, imapuser, ssl=True, imapport=993):
        self.imaphost = imaphost
        self.imapport = imapport
        self.imapuser = imapuser
        self.ssl = ssl
        self.id = imapuser + '@' + imaphost
        self.cur_uid = 0
        self.logger = logger


    def login(self, password):
        imapport = self.imapport
        if self.ssl == True:
            self.logger.debug('Open encrypted connection for "%s" to %s:%s.' % (self.imapuser, self.imaphost, imapport));
            connection = imaplib.IMAP4_SSL(self.imaphost, imapport)
        else:
            imapport = 143
            self.logger.warning('Open plaintext connection for "%s" to %s:%s.' % (self.imapuser, self.imaphost, imapport));
            connection = imaplib.IMAP4(self.imaphost, imapport)
        try:
            ret, status = connection.login(self.imapuser, password)
        except Exception as e:
            raise e
            #raise Exception('Login error.')
        self.connection = connection

    def getStatus(self, folder):
        """Returns a list [UIDVALIDITY, UIDNEXT of this folder"""
        qfolder = '"'+folder+'"'
        connection = self.connection
        ret = connection.select(qfolder, True)
        if ret[0] != 'OK':
            raise ConnectionError(ret[1])
        mbox_status = []
        ret, status = connection.status(qfolder, "(UIDVALIDITY)")
        status = [x.decode('utf-8') for x in status]
        if ret == 'OK':
            # re.search('\((\S+)\s+(\S+)\)',status[0]).groups()
            param, value = re.search('\((\S+)\s+(\S+)\)', status[0]).groups()
            mbox_status.append(value)
        else:
            self.logger.error(' Error detecting UIDVALIDTY of %s: %s[%s]', qfolder, ret, status)
            raise ConnectionError(status)
        ret, status = connection.status(qfolder, "(UIDNEXT)")
        status = [x.decode('utf-8') for x in status]
        if ret == 'OK':
            param, value = re.search('\((\S+)\s+(\S+)\)', status[0]).groups()
            mbox_status.append(value)
        else:
            self.logger.error(' Error detecting UIDNEXT of %s: %s[%s]', qfolder, ret, status)
            raise ConnectionError(status)
        return mbox_status


    def listFolder(self):
        connection = self.connection
        mailboxes = []
        self.logger.debug('Fetch folder list.')
        ret, retlines = connection.list()
        # list_pattern=re.compile(r'\((?P<flags>.*?)\) "(?P<delimiter>.*)" (?P<name>.*)')
        # IMAP returns: (\NoInferiors) "/" INBOX
        # UW IMAP returns: (\NoInferiors) NIL INBOX
        list_pattern = re.compile(r'\((?P<flags>.*?)\) "?(?P<delimiter>\S*)"? (?P<name>.*)')
        if ret == 'OK':
            for line in retlines:
                line = line.decode('utf-8')
                self.logger.debug('  Server return: %s', line)
                try:
                    flags, delimiter, mailbox = list_pattern.match(line).groups()
                    mailbox = mailbox.strip('"')
                    if r'\Noselect' in flags:
                        self.logger.debug(r'  Folder "%s" flagged as \Noselect.', mailbox)
                        continue
                except Exception as e:
                    self.logger.warning(' Skip folder. Odd server response: \""%s"\"' % (line))
                    continue
                mailboxes.append(mailbox)
        return mailboxes


    def fetchMails(self, inboxdir, folder, fetchfrom, container, envelope_to='', extraHeader=''):
        folder_maxfetch = 500
        qfolder = '"'+folder+'"'
        ident = ','.join(container)
        self.cur_uid = fetchfrom
        connection = self.connection
        ret, status = connection.select(qfolder, True)
        status = [x.decode('utf-8') for x in status]
        self.logger.debug("%s|Status of folder \"%s\": %s" % (ident, folder, status))
        if ret != 'OK':
            raise ConnectionError(status)
        searchstring = '(UID %s:*)' % self.cur_uid
        self.logger.debug("%s|Search messages: %s" % (ident, searchstring))
        result, data = connection.uid('SEARCH', None, searchstring)
        data = [x.decode('utf-8') for x in data]
        self.logger.debug("%s|Search result: %s/%s" % (ident, result, data))
        try:
            uidlist = data[0].split()
        except IndexError:
            # empty list
            self.logger.debug("%s|Empty search response: %s/%s" % (ident, self.imapuser, folder))
            return
        if folder_maxfetch > len(uidlist):
            fetchnum = len(uidlist)
        else:
            fetchnum = folder_maxfetch
        if len(uidlist) > 0 and int(uidlist[0]) >= int(self.cur_uid):
            # search returns at least the last uid number of the folder
            self.logger.info("%s|Copied %s of %s message(s) from %s/%s to %s" % (ident, fetchnum, len(uidlist), self.imapuser, folder, inboxdir))
        else:
            self.logger.debug("%s|No new messages in folder %s/%s" % (ident, self.imapuser, folder))
            return
        numfetch = 0
        for uid in uidlist:
            head = ''
            text = ''
            # fetch UIDs
            # PEEK does not set the \Seen flag implicitly
            #
            # Mail could have only header without body
            try:
                typ, head = connection.UID('FETCH', uid, '(BODY.PEEK[HEADER])')
            except Exception as e:
                self.logger.error('%s|Fail to fetch message header "%s/%s": %s"' % (ident, folder, uid, repr(e)))
                # mail not allowed without header, break
                raise(e)
            try:
                typ, text = connection.UID('FETCH', uid, '(BODY.PEEK[TEXT])')
            except Exception as e:
                # mail witout body allowed, warn and continue
                self.logger.warning('%s|Fail to fetch message body "%s/%s": %s"' % (ident, folder, uid, repr(e)))
            try:
                digest = hashlib.sha256()
                if head != '':
                    # Mail without body could fail on server, thus we catch exceptions and store only header
                    digest.update(head[0][1])
                if text != '':
                    digest.update(text[0][1])
                tmpfile = os.path.join(inboxdir, uid + '_' + ident + '-' + digest.hexdigest() + '.tmp')
            except Exception as e:
                self.logger.debug('%s|Cannot create filename from checksum from "%s/%s": %s"' % (ident, folder, uid, repr(e)))
                tmpfile = tempfile.mkstemp('.tmp', uid + '_' + ident + '-', inboxdir)[1]
            tmpfh = open(tmpfile, 'wb')

            try:
                for archive in container:
                    tmpfh.write(b'X-BENNO-GW: ')
                    tmpfh.write(archive.encode())
                    tmpfh.write(b'\r\n')
                if envelope_to:
                    tmpfh.write(b'X-REAL-RCPTTO: ')
                    tmpfh.write(envelope_to.encode())
                    tmpfh.write(b'\r\n')
                if extraHeader:
                    tmpfh.write(extraHeader.encode())
                    tmpfh.write(b'\r\n')
                if head != '':
                    tmpfh.write(head[0][1])
                if text != '':
                    tmpfh.write(text[0][1])
            except Exception as e:
                self.logger.error('%s|Fail to store message "%s/%s" as "%s": %s' % (ident, folder, uid, tmpfile, repr(e)))
                tmpfh.close()
                os.remove(tmpfile)
                raise(e)
            tmpfh.close()
            emlfile = re.sub('\.tmp$', '.eml', tmpfile)
            try:
                os.link(tmpfile, emlfile)
            except FileExistsError as e:
                self.logger.warn('%s|Email from %s:%s already exist: %s"' % (ident, folder, uid, emlfile))
            os.remove(tmpfile)
            numfetch = numfetch + 1
            self.cur_uid = int(uid)
            self.logger.debug('  %s|Store msg #%s: %s from %s of %s' % (ident, numfetch, emlfile, folder, self.imapuser))
            if numfetch >= folder_maxfetch:
                break

        self.logger.info('%s|%s/%s: %s messages up to UID %s loaded' % (ident, self.imapuser, folder, numfetch, uid))
        # smallest uid to fetch next: cur_uid+1
        self.cur_uid = self.cur_uid + 1


    def logout(self):
        try:
            self.connection.close()
        except imaplib.IMAP4.error:
            self.logger.warning('Error closing mailbox')
        self.connection.logout()


class MailboxStatus:
    """Stores UIDVALIDITY and UIDNEXT of the account"""
    def __init__(self,jsondata={}):
        if jsondata:                          # jsondata is not {}
            try:
                self.status=json.loads(jsondata)
            except Exception as e:
                self.status={}
        else:
            self.status={}


    def checkValidity(self,folder,uidvalidity):
        if not folder in self.status:
            self.status[folder]=[-1,1]
        if self.status[folder][0] == uidvalidity:
            return True
        else:
            return False


    def hasNewMails(self,folder,uidnext):
        if self.status[folder][1] == uidnext:
            return False
        else:
            return True


    def getNext(self,folder):
        return self.status[folder][1]


    def getStatus(self):
        return json.dumps(self.status)


    def getFolderlist(self):
        return self.status.keys()


    def getFolderstatus(self,folder):
        return self.status[folder][1]


    def setNext(self,folder,uidnext):
        self.status[folder][1]=uidnext


    def setValidity(self,folder,uidvalidity):
        self.status[folder][0]=uidvalidity


class ConnectionError(Exception):
    def __init__(self,value):
        self.value=value

    def __str__(self):
        return repr(self.value)
