#
# -*- coding: utf-8 -*-
#
# Benno MailArchiv
#
# Copyright 2018-2021 LWsystems GmbH
#
# http://www.lw-systems.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.
#
import base64
import configparser
import sys
import logging

from Benno import DB
from Benno.IMAP import MailboxStatus

logger = logging.getLogger('Benno.IMAP.User')

class UserError(Exception):
    def __init__(self, msg):
        self.msg=msg

    def __str__(self):
        return self.msg


class UniqueError(Exception):
    def __init(self):
        self.msg='Cannot identify unique user data'

    def __str__(self):
        return self.msg


class IMAPUser():

    @classmethod
    def CreateImapData(cls,config,imapuser,imaphost):
        """Create userobject by imapuser and imaphost"""
        iuser = IMAPUser(config)
        iuser.id = imapuser + '@' + imaphost
        iuser.imapuser = imapuser
        iuser.imaphost = imaphost
        cursor=iuser.db.cursor()
        iuser.load_imap(cursor)
        iuser.load_container(cursor)
        iuser.load_user(cursor)
        iuser.load_email(cursor)
        iuser.load_defaults()
        cursor.close()
        return iuser


    @classmethod
    def CreateUserid(cls,config,userid):
        """Create userobject by userid"""
        iuser = IMAPUser(config)
        sql = "SELECT imaphost FROM imapuser WHERE imapuser ='%s'"
        cursor=iuser.db.cursor()
        cursor.execute(sql % userid)
        rows = cursor.fetchall()
        cursor.close()
        if len(rows) > 1:
            raise UniqueError
        if len(rows) > 0:
            imaphost = rows[0][0]
            return IMAPUser.CreateImapData(config, userid, imaphost)
        else:
            raise UserError('User "%s" not found".' % userid)


    @classmethod
    def Delete(cls,imapuser):
        """Removes user from DB backend"""
        """Remove user data from database"""
        cursor = imapuser.db.cursor()
        imapuser.delete_imap(cursor)
        imapuser.delete_user(cursor)
        imapuser.delete_email(cursor)
        imapuser.delete_container(cursor)
        imapuser.db.commit()
        cursor.close()


    @classmethod
    def List(cls,config):
        """List all users"""

        iuser = IMAPUser(config)
        sql = "SELECT imapuser,imaphost FROM imapuser"
        cursor = iuser.db.cursor()
        userlist = []
        cursor.execute(sql)
        for row in cursor:
            try:
                userlist.append(IMAPUser.CreateImapData(config, row[0], row[1]))
            except UserError as e:
                logger.warning('Cannot read data for %s@%s: %s' % (row[0], row[1], e))
        cursor.close()
        return userlist


    def __init__(self, config, imapuser='', imaphost=''):
        self.config = config
        self.id = ''
        self.imapuser = imapuser
        self.imaphost = imaphost
        self.imapstatus = ''
        self.name = ''
        self.password = ''
        self.imappassword = ''
        self.email = []
        self.archive = []
        self.role = ''
        self.status = -1
        self.db = DB(config.get('DEFAULT','dbname'),
                     config.get('DEFAULT','dbuser'),
                     config.get('DEFAULT','dbpass'),
                     config.get('DEFAULT','dbhost'))

        if imaphost:
            self.id = imapuser + '@' + imaphost
        elif imapuser:
            self.id = imapuser
        self.load_defaults()


    def delete(self):
        """Remove user data from database"""
        cursor=self.db.cursor()
        self.delete_imap(cursor)
        self.delete_user(cursor)
        self.delete_email(cursor)
        self.delete_container(cursor)
        cursor.close()


    def load_imap(self,c):
        """Load data from imap table by imapuser and imaphost"""
        sql = "SELECT id,imapstatus,status,password FROM imapuser WHERE imapuser ='%s' AND imaphost = '%s'"
        c.execute(sql % (self.imapuser,self.imaphost))
        rows = c.fetchall()
        if len(rows) > 0:
            for row in rows:
                if self.id == '':
                    self.id = row[0]
                if self.imapstatus == '':
                    self.imapstatus = MailboxStatus(row[1])
                if self.status == -1:
                    self.status = row[2]
                if self.imappassword == '':
                    try:
                        b64str = row[3].split('_',1)[1]           # remove leading '$rsa$_'
                        self.imappassword = base64.b64decode(b64str.encode())
                    except IndexError:
                        self.imapppassword = row[3]
        else:
            raise UserError('User "%s@%s" not in table "imapuser".' % (self.imapuser, self.imaphost))


    def load_user(self,c):
        """Load data from user table"""
        sql = "SELECT name,password,archive,role FROM user WHERE id ='%s'"
        c.execute(sql % (self.id))
        rows = c.fetchall()
        if len(rows) > 0:
            for row in rows:
                if self.name == '':
                    self.name = row[0]
                if self.password == '':
                    self.password = row[1]
                if self.archive == '':
                    self.archive = row[2]
                if self.role == '':
                    self.role = row[3]
        else:
            raise UserError('User "%s" not in table "user".' % (self.id))


    def load_container(self,c):
        """Load container from container table"""
        sql = "SELECT cid,scid FROM container WHERE userid ='%s'" % (self.id)
        c.execute(sql)
        rows = c.fetchall()
        if len(rows) > 0:
            for row in rows:
                container=row[0]
                if row[1]:
                    container=container+'/'+row[1]
                self.archive.append(container)
            self.archive=sorted(set(self.archive))


    def load_email(self,c):
        """Load adresses for user from adress table"""
        try:
            sql = "SELECT address FROM address WHERE id='%s'" % self.id
            c.execute(sql)
            for row in c:
                self.email.append(row[0])
            self.email=sorted(set(self.email))
        except Exception as e:
            logging.warning('Cannot read data for %s from address table' % row[0])


    def load_defaults(self):
        """Load default settings from config file"""
        if self.name == '':
            self.name = self.id
        if self.role == '':
            self.role = self.config.get('DEFAULT', 'role')
        if self.imaphost == '':
            self.imaphost = self.config.get('DEFAULT', 'host')
        if not self.archive:
            try:
                self.archive.append(self.config.get('DEFAULT', 'archive'))
            except configparser.NoOptionError:
                pass


    def save(self):
        """Save user data and with optional password encryption"""
        if self.status == -1:
            self.status = 3
        try:
            self.save_all()
        except Exception as e:
            if e.__class__.__name__ == "IntegrityError":
                try:
                    self.delete()
                    self.save_all()
                except Exception as e:
                    self.db.rollback()
            else:
                raise e


    def save_all(self):
        """Store userdate in database"""
        cursor=self.db.cursor()
        self.save_imap(cursor)
        self.save_user(cursor)
        self.save_email(cursor)
        self.save_container(cursor)
        self.db.commit()
        cursor.close()


    def save_imap(self,c):
        pwstring = ''
        if self.imappassword:
            pwstring = '$rsa$_' + base64.b64encode(self.imappassword).decode()
        if self.imapstatus == '':
            jsimapstatus = {}
        else:
            jsimapstatus = self.imapstatus.getStatus()
        sql = "INSERT INTO imapuser (id,imapuser,imaphost,password,imapstatus,status) VALUES ('%s','%s','%s','%s','%s','%s')"
        c.execute(sql % (self.id,self.imapuser,self.imaphost,pwstring,jsimapstatus,self.status))


    def save_user(self,c):
        sql = "INSERT INTO user (id,name,password,archive,role) VALUES('%s','%s','%s','%s','%s')"
        # archive in user table is deprecated
        archives = ','.join(self.archive)
        c.execute(sql % (self.id,self.name,self.password,archives,self.role))


    def save_container(self,c):
        for container in self.archive:
            try:
                sql = "INSERT INTO container (userid,cid,scid) VALUES ('%s','%s','%s')"
                cid,scid=container.split('/')
            except ValueError:
                cid=container
                scid=''
            c.execute(sql % (self.id,cid,scid))


    def save_email(self,c):
        for addr in self.email:
            sql = "INSERT INTO address (id,address) VALUES('%s','%s')"
            c.execute(sql % (self.id, addr))


    def delete_imap(self,c):
        sql = "DELETE FROM imapuser WHERE ID = '%s'" % self.id
        c.execute(sql)


    def delete_user(self,c):
        sql = "DELETE FROM user WHERE ID = '%s'" % self.id
        c.execute(sql)


    def delete_email(self,c):
        sql = "DELETE FROM address WHERE id = '%s'" % self.id
        c.execute(sql)


    def delete_container(self,c):
        sql = "DELETE FROM container WHERE userid = '%s'" % self.id
        c.execute(sql)


    def changeStatus(self, status):
        """Change status bits of user"""
        # 0   00  - disabled
        # 1   01  - import enabled
        # 2   10  - login enabled
        # 3   11  - import + login enabled
        cursor = self.db.cursor()
        sql = "UPDATE imapuser SET status = '%s' WHERE id='%s'" % (status, self.id)
        cursor.execute(sql)
        self.db.commit()
        cursor.close()


    def saveIMAPPassword(self,password):
        """Save current IMAP folder status"""
        cursor = self.db.cursor()
        pwstring = '$rsa$_' + base64.b64encode(password).decode()
        sql = "UPDATE imapuser SET password = '%s' WHERE id='%s'" % (pwstring, self.id)
        cursor.execute(sql)
        self.db.commit()
        cursor.close()


    def updateIMAPStatus(self):
        """Save current IMAP folder status"""
        if self.imapstatus == '':
            jsimapstatus = {}
        else:
            jsimapstatus = self.imapstatus.getStatus()
        cursor = self.db.cursor()
        logger.debug('Store IMAP folder status: %s' % jsimapstatus)
        sql = "UPDATE imapuser SET imapstatus = '%s' WHERE id='%s'" % (jsimapstatus,self.id)
        cursor.execute(sql)
        self.db.commit()
        cursor.close()


    def resetMailbox(self):
        self.imapstatus = MailboxStatus()
