import sys
import gzip
import re
import hashlib


class Journalfile:
    '''Journalfile with predecessor data
      box     : containing box
      hash    : hash value of journal file
      pre     : name of predecessor file (True if not first journal in box)
      preBox  : name of predecessor box (True if first journal in box)
      preHash : hash value of predecessor file
      last    : True if last journal in repo
      file    : absolute filename of this journal file

   ** Settings
    * first journal in repo
      first = True
      boxFirst = True
      pre = ''
      preHash = ''
      preBox = ''

    * first journal in box, following in repo
      first = False
      boxFirst = True
      pre = ''
      preHash = hash value of predecessor journal
      preBox = name of predecessor box

    * journal in box
      first = False
      boxFirst = False
      pre = name of predecessor journalfile in this box
      preHash = hash value of predecessor journalfile in this box
      preBox = ''
    '''
    def __init__(self,box,journalfile):
        self.box=box
        self.file=journalfile
        self.id=''
        self.hash=''
        self.pre=''
        self.preHash=''
        self.preBox=''
        self.first=False
        self.last=False
        self.boxFirst=False

        if journalfile.endswith('current.journal'):
            self.last=True
        jf=self.__open()
        line1=jf.readline()
        line2=jf.readline()

        j1='"(.+)": Journal startet without predecessor'
        m=re.search(j1,line1)
        if m:
            self.id=m.group(1)
            self.boxFirst=True
            # second line shows if file has predecessor in another box
            j2='BOX CLOSED "(.+)": closed with hash (\S+)'
            m=re.search(j2,line2)
            if m:
                # preBox: predecessor boxname
                # preHash: hash of predecessor journalfile
                self.preBox=m.group(1)
                self.preHash=m.group(2)
            else:
                self.first=True
        jn='"(.+)": Continuing journal "(\S+)" with hash (\S+)'
        m=re.search(jn,line1)
        if m:
            self.id=m.group(1)
            self.pre=m.group(2)
            self.preHash=m.group(3)
        if not self.id:
            raise ValueError('Journal file format incorrect')
        # calculate hash of current journal
        sha=hashlib.sha256()
        sha.update(line1)
        sha.update(line2)
        for line in jf:
            sha.update(line)
        self.hash=sha.hexdigest().upper()
        jf.close()


    def journalId(self):
        return self.box.id+':'+self.id


    def getHashes(self):
        '''Returns a list of hashvalues of successful archived emails'''
        hashvalues=[]
        hashre=re.compile('^.+\(SUCCESS\) ARCHIVED\s+"(\S+)": archived')
        jf=self.__open()
        for line in jf:
            m=hashre.match(line)
            if m:
                hashvalues.append(m.group(1))
        hashvalues.reverse()
        return hashvalues 


    def __open(self):
        if self.file.endswith('.gz'):
            jf=gzip.open(self.file,'rb')
        else:
            jf=open(self.file,'rb')
        return jf


