package Benno::Mailfile;
use Digest::SHA qw(sha256);
use Benno;
use strict;

=head1 NAME

Benno::Mailfile - Class representing EML file from repo


=head1 SYNOPSIS

    use Benno::Mailfile;

    my $MailFile = new Benno::Mailfile($boxid,$filepath,@secretheader);
    open (my $fh, ">/path/to/emlfile.eml");
    print $fh $MailFile->eml
    close $fh;

    print "Checksum: ".$MailFile->checksum."\n";

=cut

=head1 METHODS

=head2 new($bennofile)

    my $MailFile = new Benno::Mailfile($boxid,$bennofile,$secretheader);

=cut
sub new
{
    my $class = shift;
    my ($boxid,$bennofile,@secretheader)  = @_;
    
    $boxid || die "Boxid not given\n";

    my ($chksumcount,$suffix) = $bennofile =~ /(\d\d)(\.gz)?$/;

    my $self = {
        box          => $boxid,
        bennofile    => $bennofile,
        chksumcount  => $chksumcount,   # counter of checksum collisions
        defective    => 0,
        sender       => [],
        recipients   => [],
        scid         => [],
        secretheader => \@secretheader,
        checksum     => '',
        mailcontent  => '',
    };
    bless $self, $class;

    $self->_init($bennofile);
    $self->{checksum} or die "MAILFILE NOT VALID\n";

    return $self;
}

=head2 eml()

    Return emlfile string with secretheaders.

=cut
sub eml
{
    my ($self) = @_;
    my $eml;

    if (grep /^X-REAL-MAILFROM$/, @{$self->{secretheader}}) {
        foreach my $sender (@{$self->{sender}}) {
            $eml .= "X-REAL-MAILFROM: $sender\n";
        }
    }
    if (grep /^X-REAL-RCPTTO$/, @{$self->{secretheader}}) {
        foreach my $recipient (@{$self->{recipients}}) {
            $eml .= "X-REAL-RCPTTO: $recipient\n";
        }
    }
    if (grep /^X-BENNO-SCID$/, @{$self->{secretheader}}) {
        foreach my $scid (@{$self->{scid}}) {
            $eml .= "X-BENNO-SCID: $scid\n";
        }
    }

    return $eml.$self->{mailcontent};
}


=head2 filename()

    Return the filename of the repo file.

=cut
sub filename
{
    my ($self) = @_;

    return $self->{bennofile};
}


=head2 checksum()

    Return the calculated checksum 

=cut
sub checksum
{
    my ($self) = @_;

    return $self->{checksum};
}


=head2 export(/export/dir/)

    Export to emlfile. Write atomic to dir.
    Filename: <boxid>:<checksum>.eml

    Returns the filename of the export file.

=cut
sub export
{
    my ($self,$exportdir) = @_;

    my $boxid       = $self->{box};
    my $filename    = $self->{bennofile};
    my $checksum    = $self->{checksum};
    my $chksumcount = $self->{chksumcount};

    my $tmpfile = $exportdir.'/'.$boxid.':'.$checksum.$chksumcount.'.tmp';
    my $emlfile = $exportdir.'/'.$boxid.':'.$checksum.$chksumcount.'.eml';

    -f $tmpfile and die "ERROR export $filename: $tmpfile exists\n";
    -f $emlfile and die "ERROR export $filename: $emlfile exists\n";
    open my $tmpfh, ">$tmpfile" or die "ERROR Cannot open tmpfile $tmpfile: $!\n";

    print $tmpfh $self->eml;

    close $tmpfh or die "Cannot close $tmpfile. $!\n";

    if (link($tmpfile, $emlfile)) {
        unlink $tmpfile;
    }
    else {
        unlink $tmpfile;
        die "ERROR Cannot link $tmpfile to $emlfile. $!\n";
    }

    return $emlfile;
}



sub _init
{
    my ($self,$filepath) = @_;

    my $benno_hash;
    my @sender_headers;
    my @recipient_headers;
    my @scid_headers;

    my $hash_re      = qr/^===== Hash:\s(.+?)\R/;
    my $sender_re    = qr/^Sender:.+?(\S.*?)\R/;
    my $recipient_re = qr/^Recipient:.+?(\S.*?)\R/;
    my $scid_re      = qr/^SCID:.+?(\S.*?)\R/;
    my $defective_re = qr/Defective: YES/;

    my $mail_scid    = qr/^X-BENNO-SCID/; # SCID inside mail (header not secret)

    my $fh;
    if ($filepath =~ /\.gz$/) {
        # Possible bug with fork, see Parallel::ForkManager
        open($fh, "<:unix:gzip", $filepath) || die "Cannot open gzipped BennoMailfile $filepath: $!\n";
    }
    else {
        open($fh, "<", $filepath) || die "Cannot open BennoMailfile $filepath: $!\n";
    }

    my $error;
    my $metadatadigest = '';
    foreach my $line (<$fh>) {
        if ($line =~ $defective_re) {
            $self->defective = 1;
        }
        if ($line =~ $sender_re)    {
            push @sender_headers, $1;
        }
        if ($line =~ $recipient_re) {
            push @recipient_headers, $1;
        }

        if ($line =~ $scid_re) {
            push @scid_headers, $1;
        }


        if ($line =~ $hash_re) {
            $metadatadigest = $1;

            foreach my $address (@sender_headers) {
                $address =~ s/[<>]//g;
                push @{$self->{sender}}, $address;
            }

            foreach my $address (@recipient_headers) {
                $address =~ s/[<>]//g;
                push @{$self->{recipients}}, $address;
            }

            foreach my $scid (@scid_headers) {
                push @{$self->{scid}}, $scid;
            }

            next;
        }
        $metadatadigest || next;

        $self->{mailcontent} .= $line;
    }

    my $calcdigest = Benno->digest($self->{mailcontent},@{$self->{secretheader}});
    if ($metadatadigest eq $calcdigest) {
        $self->{checksum} = $calcdigest;
        return;
    }

    print "DEBUG metadigest: $metadatadigest\n" if $ENV{DEBUG};
    print "DEBUG calcdigest: $calcdigest\n"     if $ENV{DEBUG};
    die "CHECKSUM MISMATCH\n";
}


### EOP ###
1;

