package Benno;
use strict;
use Digest::SHA qw(sha256);

our $VERSION = '2.10.0';


=head2 verify_journaladdress($journaladdress)

    Verifies the given journal email address and returns the GW header
    part if successful.

    $x_gwheader = Benno::verify_journaladdress($journaladdress);

=cut
sub verify_journaladdress
{
    my ($address) = @_;

    my ($subscr,$ckext,$salt) = $address =~ /^(\S+?)\+(\S+?)(\S{4})\@/;

    my $ckcmp = sprintf("%x",Benno::crc32($subscr.$salt));
    if ($ckext == $ckcmp) {
        return $subscr;  # X-BENNO-GW header value
    }
    return 0;
}


=head2 digest($filecontent,@secretheaders)

    Returns the digest of the file without secretheaders.

=cut
sub digest
{
    my ($class,$eml,@secretheader) = @_;

    my $ctx = Digest::SHA->new('256');
    my $in_header = 1;
    foreach my $line (split /^/m, $eml) {           # do not remove line endings
        if ($line =~ /^\R/) { $in_header = 0; }
        if ($in_header) {
            my ($headername) = $line =~ /^(\S+?):/;
            if (grep /^$headername$/i, @secretheader) {
                next;
            }
        }
        $ctx->add($line);
    }

    return uc $ctx->hexdigest;
}


=head2 id2box($bennoid)

    Returns the box id of the given bennoid.

=cut
sub id2box
{
    my ($class,$bennoid) = @_;

    my ($box_id) = $bennoid =~ m!^([^:]+?):!;
    $box_id or die "ERROR: incorrect boxid\n";

    return $box_id;
}


=head2 id2path($bennoid[$pathconfig])

    Returns the path representation of the given bennoid.

    $pathconfig optional (default 3:2)

=cut
sub id2path
{
    my ($class,$bennoid,$pathconfig) = @_;
    my $path;

    if ($pathconfig =~ /^3:2$/) {
        my ($box,$p1,$p2,$p3,$p4) = $bennoid =~ /^(.+?):(..)(..)(..)(.+)$/;
        $path = "$box/$p1/$p2/$p3/$p4";
    }
    elsif ($pathconfig =~ /\d:\d/) {
        die "ERROR: Invalid pathconfig $pathconfig\n";
    }
    else {
        my ($box,$p1) = $bennoid =~ /^(.+?):(.+)$/;
        $path = "$box/$p1";
    }

    return $path;
}



=head2 config($configfile)

    Read configuration from configfile.

    Returns config hash with config variables.

=cut
# Read conf file
# _very_ simple config file parser
#
# Config format:   var = val
#
sub config
{
    my ($class,$configfile) = @_;

    $configfile or $configfile = $class; # call as ->config() or ::config()
 
    my %config;
    open my $ch , $configfile or die "Cannot open $configfile: $!";
    foreach my $line (<$ch>) {
        next if $line =~ /^$/;
        next if $line =~ /^#/;
        $line =~ s/\R//g;
        my ($var,$val) = split(/=/, $line,2);
        # strip ws
        $var =~ s/\s//g;
        $val =~ s/^\s+//g;
        $val =~ s/\s+$//g;
        if ($val =~ /^\</) {  # read file content
            $val = Benno->_file_value($val);
        }

        $config{$var} = $val;
    }
    close $ch;
    return %config;
}


### crc32
sub crc32 {
    my ($input, $init_value, $polynomial) = @_;
    $init_value = 0 unless (defined $init_value);
    $polynomial = 0xedb88320 unless (defined $polynomial);

    my @lookup_table;
    for (my $i=0; $i<256; $i++) {
        my $x = $i;
        for (my $j=0; $j<8; $j++) {
            if ($x & 1) { $x = ($x >> 1) ^ $polynomial; }
            else { $x = $x >> 1; }
        }
        push @lookup_table, $x;
    }
    my $crc = $init_value ^ 0xffffffff;
    foreach my $x (unpack ('C*', $input)) {
        $crc = (($crc >> 8) & 0xffffff) ^ $lookup_table[ ($crc ^ $x) & 0xff ];
    }
    $crc = $crc ^ 0xffffffff;
    return $crc;
}


=head2 run_as($userid)

    Run command as user.

=cut
sub run_as {
    my ($package,$user) = @_;
    unless( $user ){
        print STDERR __PACKAGE__." must be imported with user to run as.\n";
        exit 1;
    }
    if (($< == 0) || (getpwuid($<) eq $user)) {
        my ($uid,$gid) = (getpwnam($user))[2,3];
        $( = $gid;
        $) = $gid;
        $> = $uid;
        $< = $uid;
    }
    else {
        print STDERR "Program must be run as $user or root.\n";
        exit 2;
    }
}


# read config value from file
sub _file_value
{ 
    my ($class,$configval) = @_;
    my $retval;

    (my $file = $configval) =~ s/^\<//;
    eval {
        local $/;
        open(my $fh, '<', $file) or die "Cannot read $file: $!\n";
        $retval = <$fh>;
        close($fh);
    };
    if ($@) {
        print STDERR "NOTE $@";
        return $configval;
    }
    chomp $retval;
    return $retval;
}   


### EOP ###
1;
