package Benno::Emlfile;
use strict;
use Scalar::Util qw(openhandle);
use IO::File;
use IO::String;
use Digest::SHA;

# Email::Simple module detects CRLF and set in all mails
# Wrapped headers LF at wrap and CRLF at end at end wich will not
# recognised by this Module

=head1 NAME

Benno::EMLfile - Class representing EML file


=head1 SYNOPSIS

    use Benno::EMLfile;

    my $MailFile = new Benno::EMLfile($filepath,[$secretheader]);

=cut

=head1 METHODS

=head2 new($filepath,[@secretheader])

    my $MailFile = new Benno::EMLfile($filepath);

    Optional array with custom secret header list.

=cut
sub new {
    my $class   = shift;
    my ($filepath,@secretheader) = @_;

    @secretheader = qw(X-BENNO-GW X-REAL-MAILFROM X-REAL-RCPTTO X-BENNO-IMPORTFROM X-BENNO-FOLDER) unless $secretheader[0];

    my $self = {
        secretheader   => \@secretheader,
        envelope_from  => '',
        envelope_to    => [],
        scid           => [],
        gw             => [],
        base_content   => '',
    };
    bless $self, $class;

    $self->_init($filepath);

    return $self;
}


=head2 base_content()

    Returns the base eml content without any secret headers

=cut
sub base_content { return $_[0]->{base_content}; }


=head2 content()

    Returns the content whith added secret headers

=cut
sub content 
{
    my ($self) = @_;

    my $eml;
    $eml .= 'X-REAL-MAILFROM: <'.$self->envelope_from.">\r\n";
    foreach my $val ($self->envelope_to) {
        $eml .= 'X-REAL-RCPTTO: <'.$val.">\r\n";
    }
    foreach my $val ($self->gw) {
        $eml .= 'X-BENNO-GW: '.$val."\r\n";
    }
    foreach my $val ($self->scid) {
        $eml .= 'X-BENNO-SCID: '.$val."\r\n";
    }

    return $eml.$self->{base_content};
}


=head2 digest()

    Returns the digest of eml file

=cut
sub digest { return $_[0]->{digest}; }


=head2 envelope_from()

    Returns the envelope sender

=cut
sub envelope_from
{
    my ($self,$envelope_from) = @_;

    $self->{envelope_from} = $envelope_from if $envelope_from;
    return $self->{envelope_from};
}


=head2 envelope_to()

    Returns array with envelope to adresses

=cut
sub envelope_to
{
    my ($self) = @_;
    wantarray or die "Invocation error. Method envelope_to() returns an array!\n";

    $self->{envelope_to}  && return @{$self->{envelope_to}};
    return ();
}


=head2 gw()

    Returns the value of the X-BENNO-GW header(s)

=cut
sub gw
{
    my ($self,@gw) = @_;
    wantarray or die "Invocation error. Method gw() returns an array!\n";

    $self->{gw}  && return @{$self->{gw}};
    return ();
}


=head2 msgid()

    Returns the message id

=cut
sub msgid { return $_[0]->{msgid}; }


=head2 scid()

    Returns the value of the SCID header(s)

=cut
sub scid
{
    my ($self,@scid) = @_;
    wantarray or die "Invocation error. Method scid() returns an array!\n";

    $self->{scid}  && return @{$self->{scid}};
    return ();
}


=head2 secretheader(@headerlist)

    Sets the list of secret headers
    Default: X-BENNO-GW X-REAL-MAILFROM X-REAL-RCPTTO X-BENNO-IMPORTFROM X-BENNO-FOLDER

=cut
sub secretheader
{
    my ($self,@secretheader) = @_;
    wantarray or die "Invocation error. Method secretheader() returns an array!\n";

    $self->{secretheader}  && return @{$self->{scid}};
    return ();
}



=head2 save($path[,$suffix])

    Saves the mail atomic at path. 
    Default suffix ".eml"

=cut
sub save
{
    my ($self,$path,$sfx) = @_;
    my $suffix = $sfx || '.eml';
    $suffix =~ s/^\.//;
    my $tmpname = $path.'/'.$self->{digest};
    $tmpname =~ s/\/\//\//g;
    $tmpname .= '.tmp';
    (my $emlname = $tmpname) =~ s/\.tmp$//;
    $emlname = $path.'/'.$self->{digest}.'.'.$suffix;
    -f $emlname and die "Cannot save $emlname: file exists\n";
    -f $tmpname and die "Cannot save $tmpname: file exists\n";
    open my $fh, '>',$tmpname or die "Cannot write $tmpname: $!\n";
    print $fh $self->content;
    close $fh or die "Cannot close $tmpname: $!\n";

    link($tmpname, $emlname) or die "Cannot link $tmpname to $emlname: $!\n";
    unlink $tmpname or die "Cannot remove $tmpname: $!\n";

    return $emlname;
}


=head2 set_envelope_from()

    Sets the envelope sender

=cut
sub set_envelope_from
{
    my ($self,$envelope_from) = @_;

    $self->{envelope_from} = $envelope_from;
}


=head2 set_envelope_to()

    Sets array with envelope to adresses

=cut
sub set_envelope_to
{
    my ($self,@envelope_to) = @_;

    $self->{envelope_to} = \@envelope_to;
}


=head2 set_gw()

    Sets the value of the X-BENNO-GW header(s)

=cut
sub set_gw
{
    my ($self,@gw) = @_;

    $self->{gw} = \@gw;
}


=head2 set_scid()

    Sets the value of the SCID header(s)

=cut
sub set_scid
{
    my ($self,@scid) = @_;

    $self->{scid} = \@scid;
}





sub _init
{
    my ($self,$filepath) = @_;

    my $fh;
    # check if filehandle given
    if (my $is_handle = openhandle($filepath)) {
        $fh = $is_handle;
    }
    elsif (-r $filepath) {
        #open my $fh, "$filepath" or die "Cannot open eml file $filepath: $!\n";
        $fh = IO::File->new($filepath,'r') or die "Cannot open eml file $filepath: $!\n";
    }
    else {
        # String split over scalar with lines may problematic with
        # secret headers, thus we use readline
        $fh = IO::String->new($filepath);
    }

    my $buffer;
    my $in_header = 1;
    while (my $line = $fh->getline) {
        if ($in_header) {
            $self->_init_header($line);
            $self->_check_secret($line) && next;
        }
        $buffer .= $line;
        if ($line =~ /^\R/) { $in_header = 0; }
    }
    undef $fh;

    if ($in_header) {
        # exit if no header found
        $ENV{DEBUG} eq 'maildata' &&  print STDERR "DATA READ: ",$self->{base_content},"\n";
        die "Maildata corrupt. Try environment variable DEBUG=maildata to see data\n";
    }

    my $ctx = Digest::SHA->new('256');
    $self->{digest} = uc $ctx->add($buffer)->hexdigest;
    $self->{base_content} = $buffer;
}

# no unfolding!
sub _init_header
{
    my ($self,$line) = @_;

    if ($line =~ /^X-REAL-MAILFROM:\s*?(\S.*?)\R/) {
        my $value = $1; $value =~ s/[<>]//g;
        $self->{envelope_from} = $value;
    }
    if ($line =~ /^X-REAL-RCPTTO:\s*?(\S.*?)\R/) {
        my $value = $1; $value =~ s/[<>]//g;
        push @{$self->{envelope_to}}, $value;
    }
    if ($line =~ /^X-BENNO-SCID:\s*?(\S.*?)\R/) {
        my $value = $1; $value =~ s/[<>]//g;
        push @{$self->{scid}}, $value;
    }
    if ($line =~ /^X-BENNO-GW:\s*?(\S.*?)\R/) {
        my $value = $1; $value =~ s/[<>]//g;
        push @{$self->{gw}}, $value;
    }
}


sub _check_secret
{
    my ($self,$line) = @_;

    my ($header,$value) = $line =~ /^(\S+?):\s*?(\S.*?)\R/;
    foreach my $secretname (@{$self->{secretheader}}) {
        if ($secretname =~ /^$header$/i) {
            return 1;
        }
    }
    return 0;
}

### EOP ###
1;
