package Benno::Import::RESTClient;
use HTTP::Tiny;
use MIME::Base64;
use Crypt::JWT qw(encode_jwt decode_jwt);

=head1 NAME

Benno::Import::RESTClient - Rest Import Client

=head1 SYNOPSIS

    use Benno::Import::RESTClient;

    my $RC = Benno::Import::RESTClient->new($uri[, verify_SSL => 0, ...]);
    $RC->auth_basic($username,$password);

=cut

=head1 METHODS

=head2 new($uri,{noverifycert => 0, DEBUG => 0})

    - uri - REST endpoint uri
    - params
      - noverifycert => 1
      - DEBUG => 1

=cut
sub new {
    my $class = shift;
    my ($uri,$vars) = @_;

    my $UA;
    if ($vars->{noverifycert}) {
        $UA = HTTP::Tiny->new(verify_SSL => 0);
    }
    else {
        $UA = HTTP::Tiny->new;
    }

    my $self = {
        uri     => $uri,
        UA      => $UA,
        DEBUG   => $vars->{DEBUG},
    };
    bless $self, $class;


    return $self;
}

=head2 auth_basic($username,$password)

Sets HTTP BASIC AUTH credentials.

Note: If $uri contains HTTP BASIC AUTH information, this credentials
would be skipped.

=cut
sub auth_basic
{
    my ($self,$username,$password) = @_;

    if ($uri !~ /^\@/) {    # uri auth overwrites authstring
        $self->{authstring} = 'Basic '.encode_base64($username.':'.$password,'');    # '': suppress newline at end
    }
}


=head2 auth_bearer($jwt_secret[,$payload])

Sets authentication bearer with $token.

=cut
sub auth_bearer
{
    my ($self,$jwt_secret,$payload) = @_;

    my $algo;
    $algo = 'HS256' if length($jwt_secret) >= 32;
    $algo = 'HS384' if length($jwt_secret) >= 48;
    $algo = 'HS512' if length($jwt_secret) >= 64;
    die "Shared secret must have 32 chars or more\n" unless $algo;

    my $data    = {
        sub => $payload or 'Benno MailArchiv endpoint access token',
    };

    my $jws_token = encode_jwt(payload=>$data, alg=>$algo, key=>$jwt_secret, relative_exp=>86400);

    $self->{authstring} = 'Bearer '.$jws_token;
}


=head2 upload_mail($content)

Upload mail content to Import REST interface.

=cut
sub upload_mail
{
    my ($self,$content) = @_;

    my $response,
    my $options;
    my $headers;

    if ($uri !~ /^\@/) {    # uri auth overwrites authstring
        $headers->{Authorization}  = $self->{authstring} if $self->{authstring};
    }

    $options->{headers} = $headers;
    $options->{content} = $content;

    $response = $self->{UA}->request('PUT', $self->{uri}, $options);

    my $return;
    if ($response->{success}) {
        $return = $response->{content};
        $return =~ s/\r//g;
        if ($return =~ /^OK\s/) {
            return $return;
        }
        else {
            die "Error decoding return value: $return";
        }

    }
    else {
        my $msg = 'ERROR '.$response->{status}.' '.$response->{reason};
        $msg .= ": ".$response->{content} if $response->{content};
        $msg .= "\n";
        $msg =~ s/\R+$/\n/g;
        die $msg;
    }
}

1; ### EOP ###
