package Benno::Journalfile;
use MIME::Base64;
use Benno::Emlfile;
use strict;

use base 'Benno::Emlfile';

=head1 NAME

Benno::Journalfile - Process MS or Benno Journalfiles

=head1 SYNOPSIS

    use Benno::Journalfile;
    my $Journalfile = new Benno::Journalfile($filepath);

    Inherits from Benno::Emlfile.

=cut

=head1 METHODS

=head2 new($journalfile)

    Initialize object

=cut
sub new {
    my $class   = shift;
    my ($filepath) = @_;

    # set default secret header
    my $self = $class->SUPER::new($filepath);
    $self->_journaldata();

    return $self;
}


sub _journaldata
{
    my ($self) = @_;

    require MIME::Parser;
    import MIME::Parser;

    my $Parser = new MIME::Parser;
    $Parser->output_to_core(1);
    $Parser->extract_nested_messages(0);
    my $Entity = $Parser->parse_data($self->content);
    # check journaling header
    defined($Entity->head->get('X-MS-Journal-Report')) 
    || defined($Entity->head->get('X-BENNO-JOURNALREPORT'))
    || die "Email is not a MS-Journal-Report\n";

    # journal mail structure of Exchange 2007 / 2010
    #
    # layer | type                      |
    # ------+---------------------------+----------------------------
    # 0     | multipart/mixed           |
    # 1     |     text/plain            | envelope information (text || base64)
    # 1     |     message/rfc822        | original email

    # first part contains metadata
    my $MetaData = $Entity->parts(0)         or die "Mail does not look like a journalmail.\n";
    my $BodyHandle = $MetaData->bodyhandle() or die "Mail does not look like a journalmail.\n";

    my ($sender,$senderdata,$onbehalfdata);
    my (@recipients,@scid,@gw);

    foreach my $line ($BodyHandle->as_lines) {
        $line =~ s/[\r\n]//g;
        my ($header,$value) = $line =~ /^(\S+):\s+(.+)$/i;

        if ($header =~ /^X-BENNO-GW/i)   { push @gw,   $value; next; }
        if ($header =~ /^X-BENNO-SCID/i) { push @scid, $value; next; }

        if ($header =~ /^(\x{ef}\x{bb}\x{bf})?(Recipient|To|Cc|Bcc)$/i) { # FS#1218
            foreach my $token (split /,?\s+/, $value) {                   # FS#600
                next if $token =~ /Expanded:/i;
                next if $token =~ /Forwarded:/i;
                push @recipients, $token;
            }
            next;
        }
        if ($header =~ /^(=EF=BB=BF)?Sender$/i)                { $senderdata = $value; }
        if ($header =~ /^(\x{ef}\x{bb}\x{bf})?Sender$/i)       { $senderdata = $value; } # FS#1218
        if ($header =~ /^(=EF=BB=BF)?On-Behalf-Of$/i)          { $onbehalfdata = $value; }
        if ($header =~ /^(\x{ef}\x{bb}\x{bf})?On-Behalf-Of$/i) { $onbehalfdata = $value; } # FS#1218

        if ($onbehalfdata) {
            $sender = $onbehalfdata;
        }
        else {
            $sender = $senderdata;
        }
        if ($sender =~ /\@/) {
            $sender =~ s/\<^smtp:/</;
        }

        next;
    }
    unless ($sender || @recipients) {
        die "No sender information in metadata.\n";
    }

    $self->{base_content}  = $Entity->parts(1)->stringify_body;
    $self->{envelope_from} = $sender;
    $self->{envelope_to}   = \@recipients;
    $self->{gw}            = \@gw;
    $self->{scid}          = \@scid;
}

### EOP ###
1;

