package Benno::Mailfile;
use strict;

=head1 NAME

Benno::Mailfile - Class representing EML file from repo


=head1 SYNOPSIS

    use Benno::Mailfile;

    my $MailFile = new Benno::Mailfile($filepath);
    open (my $fh, ">/path/to/emlfile.eml");
    print $fh $MailFile->eml
    close $fh;

    print "Checksum: ".$MailFile->checksum."\n";

=cut

=head1 METHODS

=head2 new($bennofile)

    my $MailFile = new Benno::Mailfile($bennofile,$boxid);

=cut
sub new
{
    my $class = shift;
    my ($boxid,$bennofile)  = @_;
    
    $boxid || die "Boxid not given\n";

    my ($chksumcount,$suffix) = $bennofile =~ /(\d\d)(\.gz)?$/;

    my $self = {
        box         => $boxid,
        bennofile   => $bennofile,
        chksumcount => $chksumcount,   # counter of checksum collisions
        defective   => 0,
        sender      => [],
        recipients  => [],
        scid        => [],
        scid_secret => 1,       # SCID not secret header
        checksum    => '',
        mailcontent => '',
    };
    bless $self, $class;

    $self->_init($bennofile);
    $self->{checksum} or die "NOT VALID\n";

    return $self;
}

=head2 eml()

    Return emlfile string with X-REAL-* Headers and mail content.

=cut
sub eml
{
    my ($self) = @_;

    my $eml;
    foreach my $sender (@{$self->{sender}}) {
        $eml .= "X-REAL-MAILFROM: $sender\r\n";
    }
    foreach my $recipient (@{$self->{recipients}}) {
        $eml .= "X-REAL-RCPTTO: $recipient\r\n";
    }

    if ($self->{scid_secret}) {
        foreach my $scid (@{$self->{scid}}) {
            $eml .= "X-BENNO-SCID: $scid\r\n";
        }
    }

    return $eml.$self->{mailcontent};
}


=head2 filename()

    Return the filename of the repo file.

=cut
sub filename
{
    my ($self) = @_;

    return $self->{bennofile};
}


=head2 checksum()

    Return the checksum of the file.

=cut
sub checksum
{
    my ($self) = @_;

    return $self->{checksum};
}


=head2 export(/export/dir/)

    Export to emlfile. Write atomic to dir.
    Filename: <boxid>:<checksum>.eml

    Returns the filename of the export file.

=cut
sub export
{
    my ($self,$exportdir) = @_;

    my $boxid       = $self->{box};
    my $filename    = $self->{bennofile};
    my $checksum    = $self->{checksum};
    my $chksumcount = $self->{chksumcount};

    my $tmpfile = $exportdir.'/'.$boxid.':'.$checksum.$chksumcount.'.tmp';
    my $emlfile = $exportdir.'/'.$boxid.':'.$checksum.$chksumcount.'.eml';

    -f $tmpfile and die "ERROR export $filename: $tmpfile exists\n";
    -f $emlfile and die "ERROR export $filename: $emlfile exists\n";
    open my $tmpfh, ">$tmpfile" or die "ERROR Cannot open tmpfile $tmpfile: $!\n";

    print $tmpfh $self->eml;

    close $tmpfh or die print STDERR "Cannot close $tmpfile. $!\n";

    if (link($tmpfile, $emlfile)) {
        unlink $tmpfile;
    }
    else {
        unlink $tmpfile;
        die "ERROR Cannot link $tmpfile to $emlfile. $!\n";
    }

    return $emlfile;
}



sub _init
{
    my ($self,$filepath) = @_;

    my $benno_hash;
    my @sender_headers;
    my @recipient_headers;
    my @scid_headers;

    my $hash_re      = qr/^===== Hash:\s(.+?)\R/;
    my $sender_re    = qr/^Sender:.+?(\S.*?)\R/;
    my $recipient_re = qr/^Recipient:.+?(\S.*?)\R/;
    my $scid_re      = qr/^SCID:.+?(\S.*?)\R/;
    my $defective_re = qr/Defective: YES/;

    my $mail_scid    = qr/^X-BENNO-SCID/; # SCID inside mail (header not secret)

    my $fh;
    if ($filepath =~ /\.gz$/) {
        # Possible bug with fork, see Parallel::ForkManager
        open($fh, "<:unix:gzip", $filepath) || die "Cannot open gzipped BennoMailfile $filepath: $!\n";
    }
    else {
        open($fh, "<", $filepath) || die "Cannot open BennoMailfile $filepath: $!\n";
    }

    my $error;
    foreach my $line (<$fh>) {
        if ($line =~ $defective_re) {
            $self->defective = 1;
        }
        if ($line =~ $sender_re)    {
            push @sender_headers, $1;
        }
        if ($line =~ $recipient_re) {
            push @recipient_headers, $1;
        }

        if ($line =~ $scid_re) {
            push @scid_headers, $1;
        }


        if ($line =~ $hash_re) {
            $self->{checksum} = $1;

            foreach my $address (@sender_headers) {
                $address =~ s/[<>]//g;
                push @{$self->{sender}}, $address;
            }

            foreach my $address (@recipient_headers) {
                $address =~ s/[<>]//g;
                push @{$self->{recipients}}, $address;
            }

            foreach my $scid (@scid_headers) {
                push @{$self->{scid}}, $scid;
            }

            next;
        }
        $self->{checksum} || next;

        if ($line =~ $mail_scid) {  $self->{scid_secret} = 0; }
        $self->{mailcontent} .= $line;
    }
}


### EOP ###
1;

