package Benno::Boxstate;
use XML::LibXML;
use Benno::Box;
use strict;


=head1 SYNOPSIS

    use Benno::Boxstate;

    my $Boxstate = new Benno::Boxstate($boxfile);

=cut

=head1 METHODS

=head2 new($bennofile)

    my $Boxstate= new Benno::Boxstate($boxfile);

=head3 config parameter

  * state
  * driver
      * fs
        * subdirs
        * dirlength
        * compression
        * storagepath
      * s3
        * endpoint
        * authKeyId
        * authKey
        * boxName
        * bennoPath
        * bucket

=cut
sub new
{
    my $class = shift;
    my ($filename)  = @_;

    my $self = {
        filename => $filename,
        boxes => {},
    };
    bless $self, $class;
    
    $self->_init($self);

    return $self;
}

=head2 add_box($Box)

    my $Boxstate->add_box($Box)

=cut
sub add_box
{
    my ($self,$Box) = @_;

    my $boxid = $Box->id;
    $self->{boxes}->{$boxid} = $Box;
}


=head2 get_box($boxid)

    my $Boxstate->get_box($boxid,)

=cut
sub get_box
{
    my ($self,$boxid) = @_;

    return $self->{boxes}->{$boxid};
}


=head2 get_boxes()

    Returns sorted list of boxes in boxstate.

    my @boxlist = $Boxstate->get_boxes()

=cut
sub get_boxes
{
    my ($self) = @_;

    my @retlist;
    foreach my $boxid (sort keys %{$self->{boxes}}) {
        push @retlist, $self->{boxes}->{$boxid};
    }

    return @retlist;
}


=head2 get_open()

    Returns the currently open box.

    my $Box = $Boxstate->get_open();

=cut
sub get_open
{
    my ($self) = @_;

    foreach my $boxid (keys %{$self->{boxes}}) {
        if ($self->{boxes}->{$boxid}->get_config('state') eq 'OPEN') {
            return $self->{boxes}->{$boxid};
        }
    }

    return undef;
}


=head2 repodir()

    my $repodir = $Boxstate->repodir()

=cut
sub repodir
{
    my ($self) = @_;

    my ($repodir) = $self->{filename} =~ m!(.*?)/[^/]+?$!;

    return $repodir;
}


sub _init
{
    my ($self) = @_;

    my $dom;
    eval {
        $dom = XML::LibXML->load_xml(location => $self->{filename});
    };
    if ($@) {
        die "Cannot parse $self->{filename}: $!\n";
    }

    # load boxes
    my $boxes;
    foreach my $node ($dom->findnodes('/boxstate/box')) {
        my $Box = new Benno::Box($node->getAttribute('identifier'));

        my $state  = $node->findnodes('box')->shift()->getAttribute('state');
        $Box->set_config('state',$state);

        my $boxconf = $node->findnodes('box/config')->shift();
        $Box->set_config('compression',$boxconf->findvalue('compression'));
        $Box->set_config('storagepath',$boxconf->findvalue('storagepath'));

        my $storage = $node->findnodes('box/config/storage')->shift();

        my $driver  = $storage->getAttribute('driver');
        $Box->set_config('driver',$driver);

        if ($driver =~ /^s3/i) {
            my $endpoint = $boxconf->findvalue('storage/endpoint');
            $endpoint =~ s!^https://!!;
            $endpoint =~ s!/.*$!!;
            $Box->set_config('endpoint',$endpoint);
            $Box->set_config('authKeyId',$boxconf->findvalue('storage/authentication/authKeyId'));
            $Box->set_config('authKey',$boxconf->findvalue('storage/authentication/authKey'));
            $Box->set_config('boxName',$boxconf->findvalue('storage/boxName'));
            $Box->set_config('bennoPath',$boxconf->findvalue('storage/bennoPath'));
            $Box->set_config('bucket',$boxconf->findvalue('storage/bucket'));
            $Box->set_config('compression',$boxconf->findvalue('compression'));
        }
        else {
            $Box->set_config('subdirs',$boxconf->findvalue('storage/subdirs'));
            $Box->set_config('dirlength',$boxconf->findvalue('storage/dirlength'));

        }
        $self->add_box($Box);
    }
}

### EOP ###
1;

