<?php
/*
 * Benno MailArchiv
 *
 * Copyright  2008-2012 LWsystems GmbH & Co. KG
 *
 * http://www.lw-systems.de/
 * http://www.benno-mailarchiv.de/
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation.
 *
 * Binary versions of this file provided by LWsystems to you as
 * well as other copyrighted, protected or trademarked materials like
 * logos, graphics, fonts, specific documentations and configurations,
 * cryptographic keys etc. are subject to a license agreement between
 * you and LWsystems.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of the GNU Affero General Public License at
 * this URI:
 * http://www.gnu.org/licenses/agpl-3.0.html
 * If not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA  02111-1307  USA
 */

/**
 $Id: DBAccess.php 27:516c6f1d1ed6 2012-02-19 15:01 +0100 mw $
*/

require_once 'DB.php';

/**
 * @package Model
 * @subpackage Persistence
 */
class DBAccess
{
    /**
     *
     * DBAccess
     *
     * Database Handling
     *
     * Configuration will be done in "config.inc.php" where the variables
     *
     * $dbUser
     * $dbPassword
     * $host
     * $database
     *
     * must be set to connect to the database. After connect, the database
     * connection object will be stored at the global var $DB.
     *
     */

    /**
     * Fehlerkennzeichnung
     *
     * Enthaelt einen wahren Wert, wenn die Datenbankabfrage einen
     * Fehler zurueckgibt.
     *
     * @access public
     */
    var $error;

    /**
     * Fehlerverhalten
     *
     * Wenn "dieError" auf TRUE gesetzt ist, beendet sich die
     * Anwendung mit die() wenn bei einer Datenbankaktion ein Fehler
     * aufgetreten ist. Transaktionen werden zurueck gesetzt.
     *
     * Defaultwert: TRUE
     *
     * @access public
     */
    var $dieError;


    /**
     * Datenbankobjekt
     *
     * Speichert das PEAR::DB Objekt.
     *
     * @access public
     */
    var $db;


    /**
     * Constructor
     *
	 * If the sqlite db is used, the $dbType parameter contains the sqlite
     * connect string:
     *
     *    sqlite:///path/to/sqlite/file
     *
     * @access public
     * @param string $dbType 
     * @param string $dbUser (optional)
     * @param string $dbPassword (optional)
     * @param string $host (optional)
     * @param string $database (optional)
     */
    function DBAccess ($dbType,$dbUser='',$dbPassword='',$host='',$database='')
    {

        $this->error = FALSE;
        $this->dieError = TRUE;

        $this->db_transaction = TRUE;
        if (strpos($dbType,'sqlite') === FALSE) {
            // if not sqlite
            $options = array('debug' => 2);
            $dns = "$dbType://$dbUser:$dbPassword@$host/$database";
            $this->db = &DB::connect($dns,$options);
        }
        else {
            $this->db = &DB::connect($dbType);
            // The sqlite library returns arrays with table.attribute
            // as key as default. This pragma forces the lib to return
            // only the attribute names.
            if (! DB::isError($this->db)) {
                $this->db->query('PRAGMA short_column_names = 1');
            }
        }
        if (DB::isError($this->db)) {
            $this->error = TRUE;
            echo " Verbindung zur Datenbank gescheitert.\n";
            error_log(' Standard Message: '.$this->db->getMessage()."\n");
            error_log(' DBMS/User Message: '.$this->db->getUserInfo()."\n");
            error_log(' DBMS/Debug Message: '.$this->db->getDebugInfo()."\n");
            die();
        }

    }


    /**
     * SQL query
     *
     * Wrapper for SQL queries. If error, the method will log this to
     * error log and triggers a rollback. The query will alway done
     * with FetchMode "DB_FETCHMODE_ASSOC", thus we will fetch an
     * associative array from the DB_result object.
     *
     * @access public
     * @param string SQL query string
     * @param bool LOG_ERROR (default=true)
     * @return object PEAR DB_result object
     */
    function query ($sql,$LOG_ERROR=true)
    {
        $this->db->setFetchMode(DB_FETCHMODE_ASSOC);
        $result =& $this->db->query($sql);
        if (DB::isError($result)) {
            $this->error = TRUE;
            if ($LOG_ERROR == true) {
                error_log(' DBMS/Debug Message: ' . $result->getDebugInfo() . '\n');
            }
        }
        return $result;
    }

    
    /**
     * fetchRow
     *
     * Fetch result row from a query directly.
     * Returns the row or an empty value.
     *
     * @access public
     * @param $sql string
     * @return DB_row object
     */
    function fetchRow ($sql)
    {
        if (preg_match('/^(INSERT|UPDATE|DELETE)/i' ,$sql)) {
            print "AFF: ".$this->db->affectedRows()."\n";
            return FALSE;
        }
        $this->db->setFetchMode(DB_FETCHMODE_ASSOC);
        $result =& $this->db->query($sql);
        if (DB::isError($result)) {
            $this->error = TRUE;
            error_log(' DBMS/Debug Message: ' . $result->getDebugInfo() . '\n');
			return FALSE;
        }
		return $result->fetchRow();
    }


    /**
     * start transaction
     *
     * @access public
     */
    function startTransaction ()
    {
        if ($this->db_transaction) {
            $this->error = FALSE;
            $this->db->query('BEGIN');
        }

    }


    /**
     * ends transaction
     *
	 * If the transaction was succesful, the method returns TRUE, otherwise
     * FALSE.
     *
     * @access public
     * @return TRUE | FALSE
     */
    function endTransaction ()
    {
        $retval = TRUE;
        if ($this->db_transaction) {
            if ($this->error) {
                $this->db->query('ROLLBACK');
                $retval = FALSE;
            }
            else {
                $this->db->query('COMMIT');
            }
            $this->error = FALSE;
        }
        return $retval;
    }

    /**
     * Transaktion zuruecksetzen
     *
     * Fuehrt ein 'ROLLBACK' aus, 'db_transaction' auf einen wahren Wert
     * gesetzt ist.
     *
     * @access public
     */
    function rollback ()
    {
        if ($this->db_transaction) {
            $this->db->query('ROLLBACK');
        }
    }

    /**
     * Quote smart
     *
     * @access public
     * @param mixed
     * @return mixed
     */
    function quote ($str)
    {
        //return $this->db->quoteSmart($str);
        return $this->db->escapeSimple($str);
    }


    /**
     * Returns the number of affected rows
     *
     * Need for INSERT, UPDATE or DELETE queries.
     *
     * Not supported by all database backends!
     *
     * @access public
     * @return mixed
     */
    function affectedRows ()
    {
        return $this->db->affectedRows();
    }

}

//$DB = new DBAccess($dbType,$dbUser,$dbPassword,$host,$database);
//return $DB;

?>
