<?php
/*
 * Benno MailArchiv
 *
 * Copyright  2008-2012 LWsystems GmbH & Co. KG
 *
 * http://www.lw-systems.de/
 * http://www.benno-mailarchiv.de/
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation.
 *
 * Binary versions of this file provided by LWsystems to you as
 * well as other copyrighted, protected or trademarked materials like
 * logos, graphics, fonts, specific documentations and configurations,
 * cryptographic keys etc. are subject to a license agreement between
 * you and LWsystems.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of the GNU Affero General Public License at
 * this URI:
 * http://www.gnu.org/licenses/agpl-3.0.html
 * If not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA  02111-1307  USA
 */

/** $Id: Persistence.php 27:516c6f1d1ed6 2012-02-19 15:01 +0100 mw $
 *
 * Copyright (c) 2005 Martin Werthmoeller <mw@werthmoeller.de>
 *
 */
/**
 * @package Persistence
 */
/**
 * 
 * Persistence layer
 *
 * Abstract class to inherit
 *
 * The table names must be lowercase at the database.
 *
 */
require_once 'SPAF2/DBAccess.php';

class Persistence
{
    /**
     * The id of the object
     *
     * The id parameter must be set by the class. It must be uniq and
     * will be the primary key at the database.
     *
     * @access public
     * @var string
     */
    var $id;


    /**
     * Class method to fetch a list of object ids
     *
     * The optional parameter order can be used to set an order
     * column. Returns a list of object ids.
     *
     * @access public
     * @param string $class
     * @param string $order (optional)
     * @return array entries
     */
    function IdList ($class,$order='')
    {
        global $App;

        $class = strtolower($class);
        if ($order) {
            // '$order' without quotes -> sqlite does not like it
            $sql = "SELECT id,title FROM $class ORDER BY $order";
        }
        else {
            $sql = "SELECT id FROM $class";
        }
        $res = $App->DB->query($sql);
        $idlist = array();
        while ($row = $res->fetchRow()) {
            array_push($idlist, $row['id']);
        }
        return $idlist;
    }


    /**
     * Class method to fetch a list of objectdata
     *
     * The optional parameter order can be used to set an order
     * column. Returns a list of object ids.
     *
     * This method returns an array with the data of all objects to
     * create a bulk of object with one database call. The returned
     * array is an array of arrays like like:
     *
     *     [
     *       [  id => $id1, attr11 => $val11, attr12 => $val12, ... ]
     *       [  id => $id2, attr21 => $val21, attr22 => $val22, ... ]
     *     ]
     *
     * The calling method can use this array to construct the objects.
     *
     * @access public
     * @param string $class
     * @param string $order (optional)
     * @return array entries
     */
    function OdList ($class,$order='')
    {
        global $App;
        $class = strtolower($class);
        if ($order) {
            // '$order' without quotes -> sqlite does not like it
            $sql = "SELECT * FROM $class ORDER BY $order";
        }
        else {
            $sql = "SELECT * FROM $class";
        }
        $res = $App->DB->query($sql);
        $odlist = array();
        while ($row = $res->fetchRow()) {
            array_push($odlist, $row);
        }
        return $odlist;
    }


    /**
     * Load data from database to object
     *
	 * If the parameter $attributes not given, the method tries to load all
     * attributes of the object directly from the database.
     *
     * @access public
     * @param string id
     * @param array $attributes (optional)
     * @return object | FALSE
     */
    function load ($attributes='')
    {
        global $App;
        global $DEBUG;
        $DB = $App->DB;

        $tablename = $this->_tablename();

        if ($attributes) {
            $primary_key = $attributes[0]; // first attribute is key
            $primary_val = $this->$primary_key;
        }
        else {
            $primary_key = 'id';
            $primary_val = $this->id;
        }

        if (! $primary_val)
            error_log("[SPAF2/Persistence] Attribute 'id' not set in object.");

        $sql = "SELECT * FROM $tablename WHERE $primary_key = '$primary_val'";
        $res = $App->DB->query($sql);
        if ($App->DB->error) {
            error_log('[SPAF2/Persistence] DB error:'.$App->DB->message);
            return FALSE;
        }
        $row = $res->fetchRow();

        if (! $row[$primary_key]) {
            if ($DEBUG)
                error_log('[SPAF2/Persistence] Object with id \''.$primary_val.'\' not in table \''.$tablename.'\'.');
            return FALSE;
        }
        foreach ($row as $var => $val) {
            $this->$var = $val;
        }
        return $this;
    }



    /**
     * Update the object at the persistence layer
     *
     * The object will be updated at the persistence layer. If not
     * exists, the object will be created.
     *
     * Attributes that begin with an underscore '_' will not be considered.
     *
     * If the object has the id attribute not set, it will be set to a
     * uniq value.
     *
     * The method may fail, if the 'affected_rows()' is not supported
     * by the database backend api.
     *
     * If the parameter $attributes not given, the method tries to load all
     * attributes of the object directly from the database.
     *
     * The method returns TRUE or FALSE if the database transaction failed.
     *
     * @access public
     * @param array $attributes (optional)
     * @return TRUE | FALSE
     */
    function save ($attributes='')
    {
        global $App;
        $DB = $App->DB;

        $tablename = $this->_tablename();
        if ($attributes) {
            $primary_key = $attributes[0]; // first attribute is key
            $primary_val = $this->$primary_key;
            foreach ($attributes as $attr) {
                $attrs[$attr] = $this->$attr;
            }
        }
        else {
            $attrs = $this->_getAttributeList();
            if (! array_key_exists('id',$attrs)) {
                error_log("[SPAF2/Persistence] Attribute 'id' does not exist at class \'$tablename\'.");
                throw new Exception('No id in object');
            }

            $primary_key = 'id'; // first attribute is key
            $primary_val = $this->id;
            // construct id if not set
            if (! $this->id)
                $this->id = uniqid(getmypid()); 
        }


        // construct update query
        $sql_update = "UPDATE $tablename SET ";
        foreach ($attrs as $var => $val) {
            $sql_update .= $DB->quote($var)."='".$val."',";
        }
        // remove the last comma
        $sql_update = rtrim($sql_update,',');
        $sql_update .= " WHERE ".$primary_key." = '".$primary_val."';";
        

        // construct insert query
        $sql_insert = "INSERT INTO $tablename (";
        foreach ($attrs as $var => $val) { $sql_insert .= $var.','; }
        // remove the last comma
        $sql_insert = rtrim($sql_insert,',');
        $sql_insert .= ') VALUES (';
        foreach ($attrs as $var => $val) { $sql_insert .= "'".$val."',"; }
        // remove the last comma
        $sql_insert = rtrim($sql_insert,',');
        $sql_insert .= ');';

        $DB->startTransaction();
        $row = $DB->query($sql_update);
        if ($DB->affectedRows() === 0) {
            $DB->query($sql_insert);
        }

        return $DB->endTransaction();
    }


    /**
     * Remove this object.
     *
	 * Removes current object from the persistence container.
     *
     * @access public
     * @param array $attributes (optional)
     * @return int number of deleted entries
     */
    function delete ($attributes='')
    {
        global $App;
        $DB = $App->DB;

        if ($attributes) {
            $primary_key = $attributes[0]; // first attribute is key
            $primary_val = $this->$primary_key;
        }
        else {
            $primary_key = 'id';
            $primary_val = $this->id;
        }

        $tablename = $this->_tablename();
        $sql_delete = 'DELETE FROM '.$DB->quote($tablename).' WHERE '.$primary_key." = '$primary_val'";
        $row = $DB->query($sql_delete);
        return $DB->affectedRows();
    }


  /**
     * Private method to fetch the attributes of the current object.
     *
     * Attributes with a '_'-prefix will not be considered.
     *
     * @access private
     * @return array
     */
    function _getAttributeList ()
    {
        $ret = array();
        $arr = get_object_vars($this);
        foreach ($arr as $var => $val) {
            if (strpos($var,'_') === 0) continue; 
            $ret[$var] = $val;
        }
        return $ret;
    }

    /**
     * Private method to extract table name
     *
     * Table names are constructed from the class name with all
     * characters converted to lowercase.
     *
     * @access private
     * @return string $tablename
     */
    function _tablename ()
    {
        return strtolower(get_class($this));
        return get_class($this);
    }


}
?>
