<?php
/*
 * Benno MailArchiv
 *
 * Copyright  2008-2012 LWsystems GmbH & Co. KG
 *
 * http://www.lw-systems.de/
 * http://www.benno-mailarchiv.de/
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation.
 *
 * Binary versions of this file provided by LWsystems to you as
 * well as other copyrighted, protected or trademarked materials like
 * logos, graphics, fonts, specific documentations and configurations,
 * cryptographic keys etc. are subject to a license agreement between
 * you and LWsystems.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of the GNU Affero General Public License at
 * this URI:
 * http://www.gnu.org/licenses/agpl-3.0.html
 * If not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA  02111-1307  USA
 */

require_once 'model/DateConvert.php';

class Query
{

    /**
     * id
     */
    var $id;

    var $text;              // content or search term
    var $sender;
    var $recipient;
    var $subject;
    var $headerName;        // will only be set on extended search
    var $headerValue;       // will only be set on extended search
    var $isAttachment;
    var $isSpam;
    var $dateFrom;          // 01.01.1970
    var $dateTo;            // 31.12.2011

    /**
     * array with sort conditions
     *
     *   array('field' => 'Date', 'ascending' => 'false')
     */
    var $sort;
    var $displayname;       // name shown in user interface


    /**
     * Query constructor
     */
    function Query ($id,$text='',$dateFrom='',$dateTo='',$sender='',$recipient='',$subject='',$headerName='',$headerValue='',$isAttachment=false,$isSpam=false,$displayname='')
    {
        $this->id = $id;
        $this->text = $text;          // content or search term
        $this->dateFrom = $dateFrom;
        $this->dateTo = $dateTo;
        $this->sender = $sender;
        $this->recipient = $recipient;
        $this->subject = $subject;
        $this->headerName = $headerName;        // only in extended search
        $this->headerValue = $headerValue;
        $this->isAttachment = $isAttachment;
        $this->isSpam = $isSpam;
        $this->sort  = array('field' => 'Date', 'ascending' => 'false');
        $this->displayname = $displayname;
    }


    /**
     * returns formatted date
     *
     * @return string
     */
    function getDateFrom()
    {
        global $App;
        if ($this->dateFrom) {
            return strftime($App->date_format,$this->dateFrom);
        }
    }


    /**
     * returns formatted date
     *
     * @return string
     */
    function getDateTo()
    {
        global $App;
        if ($this->dateTo) {
            return strftime($App->date_format,$this->dateTo);
        }
    }


    /**
     * returns an array with query data
     *
     * @return array $query
     *
     */
    function getQuery ()
    {
        global $App;
        $query_string = '';

        if ($this->headerName) {            // every other param could be ''
            // extended query
            $query_string = $this->_extendedQuery();
        }
        else {
            $query_string = $this->_simpleQuery();
        }

        if ($App->DEBUG >= 1) {
            $App->log('[Query] '.$query_string);
        }
        // replace special chars
        $search = array('&');
        $replace = array('%26');
        // Achtung! Hier liegt ein Lucene Query-String vor!
        $query_string = str_replace($search,$replace,$query_string);

        return $query_string;
    }


    /**
     * returns string for the simple search form
     *
     * @return string
     *
     */
    function getSimple ()
    {
        global $App;

        $field_text = '';
        if (! $this->headerName) {            // every other param could be ''
            $field_text = $this->text;
        }
        return $field_text;
    }

 

    /**
     * Returns sort field setting
     */
    function getSortField()
    {
        $sortField = 'Date';
        if ($this->sort == 'dateAsc') {
            $sortField = 'Date';
        }
        elseif ($this->sort == 'subjectAsc') {
            $sortField = 'Subject';
        }
        elseif ($this->sort == 'subjectDesc') {
            $sortField = 'Subject';
        }
        return $sortField;
    }

    /**
     * Returns if sort is ascending (true) or not ascending (false)
     */
    function getSortAsc()
    {
        $sortOrder = 'false';
        if ($this->sort == 'dateAsc') {
            $sortOrder = 'true';
        }
        elseif ($this->sort == 'subjectAsc') {
            $sortOrder = 'true';
        }
        elseif ($this->sort == 'subjectDesc') {
            $sortOrder = 'false';
        }
        return $sortOrder;
    }
 

    /**
     * Query for spam mail too.
     *
     */  
    function setSpam($value=false)
    {
        $this->isSpam = $value;
    }


    /**
     * Set the sort field and order
     *
     * Possible values:
     *  - dateDesc
     *  - dateAsc
     */  
    function setSortOrder($sort='dateDesc')
    {
        $this->sort = $sort;
    }


    /**
     * simple query with only one search field
     *
     * @return string $query
     */
    function _simpleQuery ()
    {
        $fieldlist = array('Sender', 'Recipient', 'Subject', 'Text');

        if ($this->text) {
            $qstring = $this->text;
        }
        else {
            throw new Exception('QUERY_EMPTY');
        }

        $query_string = '('.$this->_format_query($qstring,$fieldlist).')';

        if (! $this->isSpam) {
            $query_string .= ' AND NOT HEADER-X-SPAM-FLAG:YES';
        }

        $date_range   = $this->_dateRange($this->dateFrom,$this->dateTo);
        $query_string .= ' AND ('.$date_range.')';

        return $query_string;

    }


    /**
     * query with AND(ed) fields
     *
     * @return string $query
     */
    function _extendedQuery ()
    {
        global $App;

        $and   = ' AND ';
        $or    = ' OR ';

        $query_string = '';

        if ($this->sender !== '') {
            $sender_q = $this->sender.'*';
            if (!$App->getConfig('SUPPRESS_LEADING_WILDCARD')) {
                $sender_q = '*'.$this->sender.'*';
            }
            $sender_string = $this->_format_query($sender_q,'Sender');
            // (!)Sender contains plain email adress without names
            $from_string   = $this->_format_query($sender_q,'From');
            $query_string .= '('.$sender_string.$or.$from_string.')';
            $query_string .= $and;
        }

        if ($this->recipient !== '') {
            $recipient_q = $this->recipient.'*';
            if (!$App->getConfig('SUPPRESS_LEADING_WILDCARD')) {
                $recipient_q = '*'.$this->recipient.'*';
            }
            // (!)Recipient contains plain email adress without names
            $recipient_string = $this->_format_query($recipient_q,'Recipient');
            $to_string        = $this->_format_query($recipient_q,'To');
            $cc_string        = $this->_format_query($recipient_q,'Cc');
            $query_string .= '('.$recipient_string.$or.
                                 $to_string.$or.
                                 $cc_string.')';
            $query_string .= $and;
        }

        if ($this->subject !== '') {
            $subject_string = $this->_format_query($this->subject,'Subject');
            $query_string .= '('.$subject_string.')';
            $query_string .= $and;
        }

        if ($this->headerValue !== '') {
            $query_string .= '('.strtoupper('HEADER-'.$this->headerName).
                             ':"'.$this->headerValue.'")';
            $query_string .= $and;
        }

        if ($this->text !== '')   {
            $text_string = $this->_format_query($this->text,'Text');
            $query_string .= '('.$text_string.')';
            $query_string .= $and;
        }

        // search for all (*) messages for date-only queries
        if ($query_string == '') {
            if (($this->dateFrom != '') || ($this->dateTo != '')) {
                $query_string = '(Subject:*)'.$and;
            }
            else {
                throw new Exception('EQUERY_EMPTY');
            }
        }

        if ($this->isAttachment) {
            $query_string .= 'hasAttachment:[1 TO 9999]'.$and;
        }

        if (!$this->isSpam) {
            // Todo customizable spam flag headers
            $query_string .= 'NOT HEADER-X-SPAM-FLAG:YES'.$and;
        }

        $date_range   = $this->_dateRange($this->dateFrom,$this->dateTo);
        $query_string .= $date_range;

        return $query_string;
    }


    /**
     * format querystring
    */
    function _format_query($qstring,$field)
    {
        global $App;

        if (is_array($field)) {
            $fieldlist = $field;
        }
        else {
            $fieldlist = array($field);
        }
        $tokenlist = $this->_tokenize_query($qstring);
        $query_string = '';
        foreach ($fieldlist as $field) {
            $field_string = $this->_format_fieldstring($field,$tokenlist);
            $query_string .= '('.$field_string.')';
        }

        if ($App->getConfig('SUPPRESS_LEADING_WILDCARD')) {
            if (preg_match('/:[\*\?]/',$query_string)) {
                $App->log('[Query] Remove leading wildcards from query.');
                throw new Exception('LEADING_WILDCARD');
            }
        }

        return $query_string;
    }

    /**
     * Tokenize querystring considering boolean prefixes (+-)
     */
    function _tokenize_query($qstring)
    {
        $tokenlist = array();
        while (strlen($qstring)) {
            $qstring = trim($qstring);
            $matches = array();

            // ^\S matches "\S, thus check literal in "..." first
            if (preg_match('/^([+-]?"[^"]+")/',$qstring,$matches)) {
                array_push($tokenlist,$matches[0]);
                $qstring = preg_replace('/^([+-]?"[^"]+")/','',$qstring);
                continue;
            }
            // match first "word" in string
            if (preg_match('/^(\S+\s*?)/',$qstring,$matches)) {
                array_push($tokenlist,$matches[0]);
                $qstring = preg_replace('/^(\S+)/','',$qstring);
                continue;
            }
        }
        return $tokenlist;
    } 


    function _format_fieldstring($field_name,$tokenlist)
    {
            // prepare for field
            $field_string = '';
            foreach ($tokenlist as $token) {
                $prefix = '';
                if (preg_match('/^([\+-])/',$token,$match)) {
                    $prefix = $match[0];
                }
                $token = preg_replace('/^[\+-]/','',$token,-1);
                $encprefix = urlencode($prefix);
                $field_string .= $encprefix.$field_name.':'.$token.' ';
            }
            $field_string = preg_replace("/\s$/",'',$field_string);

            return $field_string;
    }


    /**
     * return the lucene string for the date range
     */
    function _dateRange ($first,$last)
    {
        if (!$first) {
            $first = 0;
        }
        if (!$last) {
            $last = 2147483647;
        }
        $firstString = strftime('%Y%m%d%H%M',$first);
        $lastString  = strftime('%Y%m%d%H%M',$last);
	    $lastString  = substr($lastString, 0, -4)."2400"; 
        return "Date:[$firstString%20TO%20$lastString]";
    }

}

?>
