<?php
/*
 * Benno MailArchiv
 *
 * Copyright  2008-2012 LWsystems GmbH & Co. KG
 *
 * http://www.lw-systems.de/
 * http://www.benno-mailarchiv.de/
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation.
 *
 * Binary versions of this file provided by LWsystems to you as
 * well as other copyrighted, protected or trademarked materials like
 * logos, graphics, fonts, specific documentations and configurations,
 * cryptographic keys etc. are subject to a license agreement between
 * you and LWsystems.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of the GNU Affero General Public License at
 * this URI:
 * http://www.gnu.org/licenses/agpl-3.0.html
 * If not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA  02111-1307  USA
 */

/* $Id: Application.php 27:516c6f1d1ed6 2012-02-19 15:01 +0100 mw $ */
/**
 * @package Application
 */
/**
 * Application base class
 *
 * The applciation class uses special session variables. The following
 * session variables are reserved:
 *
 *   APP_AUTHZLEVEL
 *
 */
@include_once 'lib/Smarty.class.php';

require_once 'SPAF2/ApplicationConfig.php';
require_once 'SPAF2/DBAccess.php';
require_once 'SPAF2/AppUtils.php';

class Application extends ApplicationConfig
{
    /**
     * _authz | authz
     *
     * Array with authorization mappings.
     *
     * @access public
     * @var array
     */
    var $_authz;
    var $authz;


    /**
     * array with config parameters
     *
     * @access public
     * @var array
     */
    var $config;


    /**
     * ApplicationError object
     *
     * @access public
     * @var object ApplicationError
     */
    var $Error;

    /**
     * _output
     *
     * Holds the output string.
     *
     * @access public
     */
    var $_output;

    /**
     * _request
     *
     * Array with request data.
     *
     * @access public
     * @var array
     */
    var $_request;

    /**
     * module
     *
     * The name of the current module or '.' if main module.
     *
     * @access public
     * @var string
     */
    var $module;


    /**
     * DEBUG
     *
     * @access public
     * @var string
     */
    var $DEBUG;

    /**
     * configfile
     *
     * Main configuration file of this application.
     *
     * @access public
     * @var string
     */
    var $configfile;

    /**
     * configdir
     *
     * The configuration directory of this application is the
     * dirname() of the configuration file.
     *
     * @access public
     * @var string
     */
    var $configdir;

    /**
     * _session
     *
     * Array with session data.
     *
     * @access public
     * @var array
     */
    var $_session;

    /**
     * _vars
     *
     * Array with application variables.
     *
     * @access public
     * @var array
     */
    var $_vars;

    /**
     * _debug_var
     *
     * Holds variables to debug.
     *
     * @var array
     * @access public
     */
    var $_debug_vars;

    /**
     * _TPL
     *
     * Holds the templacte engine class.
     *
     * @access public
     */
    var $TPL;


    /**
     * skin
     *
     * Holds the name of the skin subdir
     *
     * @access public
     */
    var $skin;


    /**
     * lang
     *
     * Holds the name of the language setting
     *
     * @access public
     */
    var $lang;


    /**
     * date_format
     *
     * Holds the current date format 
     *
     * @access public
     */
    var $date_format;


    /**
     * template
     *
     * Holds the current templacte to render.
     *
     * @access public
     */
    var $template;


    /**
     * message_file
     *
     * Filename of the message file
     *
     * @access public
     */
    var $message_file;


    /**
     * view
     *
     * The current view.
     *
     * @access public
     */
    var $view;


    /**
     * DB
     *
     * If set, holds the DBAccess object of the current application.
     *
     * @access public
     */
    var $DB;


    /**
     * constructor
     *
     * The "configfile" points to current configfile.
     *
     * The Constructor will access the config files 'application.cfg'
     * and 'authz.cfg'.
     *
     * @access public
     * @param string $configfile
     * @param array $_REQUEST request parameters
     * @return object
     */
    function Application ($configfile,$request,$MODULE)
    {

        $this->_debug_vars = array();
        $this->configfile = basename($configfile);
        $this->configdir = dirname($configfile);

        $AppConfig = new ApplicationConfig($this->configdir);
        $this->config = $AppConfig->load($this->configfile);
        $this->_authz = $AppConfig->load($this->config['AUTHORIZATION']);
        $this->authz  = $this->_authz;

        $this->module   = $MODULE;
        $this->_request = & $_REQUEST;
        $this->_session = & $_SESSION;
        $this->DEBUG = $this->getConfig('DEBUG');

        // Language settings
        $default_locale = $this->getConfig('DEFAULT_LANGUAGE');
        if ($this->getParam('LANG')) {
            $this->lang = $this->getParam('LANG');
        }
        elseif ($this->getSession('LANG')) {
            $this->lang = $this->getSession('LANG');
        }
        else {
            $this->lang = AppUtils::GetLocaleFromBrowser($default_locale, false);
        }
        $this->setSession('LANG',$this->lang);

        // date format
        switch ($this->lang) {
            case 'de':
                $this->date_format = '%d.%m.%Y';
                break;
            case 'en':
                $this->date_format = '%m/%d/%Y';
                break;
        }


        if ($this->getParam('SKIN')) {
            $this->skin = $this->getParam('SKIN');
        }
        elseif ($this->getSession('SKIN')) {
            $this->skin = $this->getSession('SKIN');
        }

        // Template initialization
        $this->TPL = new Smarty;
        $this->TPL->compile_check = true;

        if(array_key_exists('TEMPLATE_ROOT',$this->config)) {
            $tpl_root = getcwd().'/'.$this->config['TEMPLATE_ROOT'].'/templates';
        }
        else {
            $tpl_root = getcwd().'/templates';
        }

        if ($this->skin) {
            $skin_dir = $tpl_root.'/'.$this->skin;
        }
        // fallback
        if (! is_file($skin_dir.'/header.tpl')) {
            $skin_dir = $tpl_root.'/default';
        }
        $this->TPL->template_dir = $skin_dir;

        $message_file = $tpl_root.'/'.$this->lang.'.msg';
        if (! is_readable($message_file)) {
            $message_file = $tpl_root.'/'.$default_locale.'.msg';
        }

        // skin localization
        $locale_text = $tpl_root.'/'.$this->lang.'.txt';
        if (! is_readable($locale_text)) {
            $locale_text = $tpl_root.'/'.$default_locale.'.txt';
        }

        // messages from file
        $this->message_file = $message_file;
        $this->Error = new ApplicationError ($message_file);

        $this->log('template_dir '.$skin_dir,2,'SPAF2/App');
        $templates_cache_base = $this->getConfig('TEMPLATE_CACHE');
        $templates_cache_skin = $templates_cache_base.'/'.$this->skin;
        if (!is_dir($templates_cache_skin)) {
            mkdir($templates_cache_skin);
        }
        if (is_writable($templates_cache_skin)) {
            $this->TPL->compile_dir = $templates_cache_skin;
        }
        else {
            $this->log('WARNING: Cannot write to templates cache: '.$templates_cache_skin,0,'SPAF2/App');
            $this->TPL->compile_dir = '/tmp';
        }
        $this->TPL->config_dir   = $this->configdir;
        $this->TPL->cache_dir    = $tpl_root . '/cache';
        $this->TPL->assign('LOCALE_TEXT',$locale_text);
        $this->TPL->assign('SN',session_name());
        $this->TPL->assign('SID',session_id());
        $this->TPL->assign('CA',$this->getParam('CA'));
        $this->TPL->assign_by_ref('ERROR',$this->Error);

        if ($DEBUG = $this->getParam('DEBUG')) {
            $this->setSession('DEBUG',$DEBUG);
        }
        $this->TPL->assign('DEBUG',$this->DEBUG);

        if (isset($this->config['DBTYPE'])) {
            $this->DB = new DBAccess($this->config['DBTYPE'],$this->config['DBUSER'],
                                     $this->config['DBPASS'],$this->config['DBHOST'],
                                     $this->config['DATABASE']);
        }


    }


    /**
     * Assign the variable to the template
     *
     * @access public
     * @param string $name Name of the template variable
     * @param string $var  Variable to assign
     */
    function assign ($name,$var)
    {
        $this->TPL->assign($name,$var);
    }


    /**
     * authz
     *
     * Check if the current user is allowed to access the given
	 * action. Sets the error ERR_AUTHZ if the current user is not authorized
     * to access the given action.
     *
	 * If the class method Authorization::check() exists, the method check will
	 * be called with the $action variable and the current $App object.
     * The return value of this method will be returned.
	 * Otherwise the authorization will be checked against the value of the
	 * authz configuration for this action. If successful a TRUE value will be
     * returned.
     *
     * @access public
     * @param string $action
     * @return bool
     */
    function authz ($action)
    {
        $authorized = FALSE;
        if (class_exists('Authorization'))
        {
            $authorized = Authorization::check($action,$this);
        }
        elseif (isset($this->_authz[$action])) {
            if ($this->accessLevel() >= $this->_authz[$action]) {
                $authorized = TRUE;
            }
        }
        if (! $authorized) {
            $this->setError('ERR_AUTHZ');
        }
        return $authorized;
    }


    /**
     * Return the value of a configuration parameter.
     *
     * @access public
     * @param string $parameter
     * @param string $default (optional)
     * @return misc $value
     */
    function getConfig($parameter,$default='')
    {
        if (isset($this->config[$parameter])) {
            return  $this->config[$parameter];
        }
        else if ($default) {
            return $default;
        }
        else {
            return FALSE;
        }
    }



    /**
     * accessLevel
     *
	 * Returns the authorization level of the current user / session.
     *
     * @access public
     * @return int
     */
    function accessLevel ()
    {
        $level = 0;
        if (array_key_exists('ACCESS_LEVEL',$this->_session)) {
            $level = $this->_session['ACCESS_LEVEL'];
        }
        if (! $level) { $level = 0; }
        return $level;
    }


    /**
     * setAccessLevel
     *
	 * Sets the authorization level of the current user / session. Returns the
     * access level.
     *
     * @access public
     * @param int accesslevel
     * @return int
     */
    function setAccessLevel ($level=0)
    {
        $this->_session['ACCESS_LEVEL'] = $level;
        return $this->_session['ACCESS_LEVEL'];
    }


    /**
     * Compile the template and append to output
     *
     * The suffix of the template file (default: .tpl) must not be
     * given. If no suffix given, '.tpl' will appended.
     *
     * @access public
     * @param string $template Template filename
     */
    function compile ($template)
    {
        /* The smarty debuggig could only be used with the display
         * method.
         */
        ob_start();
        //if (file_exists(SMARTY_DIR.'/'.$template)) {
        if (file_exists($template)) {
            $this->TPL->display($template);
            $this->_output .= ob_get_contents();
        }
        elseif(@$this->config['DEBUG'] > 0) {
            $this->log("NO TEMPLATE $template",0,'SPAF2/App');
        }
        ob_end_clean();
    }


    /**
     * Calls (require) the file
     *
     * This method inculdes (require(s)) the file an sets the current
     * Application object to the local variable $App.
     *
     * @access public
     * @param string $file
     * @return misc $return return value of the included file
     */
    function call ($file)
    {
        $App =& $this;          // need by include
        return require $file;
    }


    /**
     * Log something
     *
     * @access public
     * @param string $message
     */
    function log ($message,$threshold=0,$module='')
    {
        $view = '-';
        if ($this->view != '') {
            $view = $this->view;
        }
        if ($module) {
            $view = $module;
        }
        if ($this->DEBUG >= $threshold) {
            error_log("[$threshold:$view] $message");
        }
    }


    /**
     * debug
     *
     * Perform debugging
     *
     * Shows the debugging console if the $debug parameter is set in
     * session and in configuration.
     *
     * @access public
     * @param string $debug
     */
    function debug ($debug)
    {
        $debug_val = $this->getSession('DEBUG');
        if ($debug && $debug_val) {
            $AppDebug = new ApplicationDebug($this->TPL,$this->_request,$this->_session,$this->_debug_vars);
            $this->_output .= $AppDebug->debug($debug_val);
        }
    }


    /**
     * debug_var
     *
     * Perform debugging for one variable
     *
     * Shows the debugging console with the variable.

     * @access public
     * @param misc $variable
     */
    function debug_var ($var)
    {
        array_push($this->_debug_vars,$var);
    }


    /**
     * Returns the output to the browser.
     *
     * @access public
     *
     */
    function display ()
    {
        echo $this->_output;
    }


    /**
     * getParam
     *
     * Returns the attribute string with html entities replaced.
     *
     * This method uses the stripslashes() and htmlentities() functions of PHP
     * to convert the parameter string. 
     *
     * If the parameter $raw has been set, no conversation will be performed on
     * the returned parameter.
     * 
     * If the global variable $UNICODE has been set, the htmlentities function
     * will not be performed on the parameters.
     *
     * @access public
     * @param name
     * @param bool $raw to fetch raw data
     * @return string
     */
    function getParam($attr,$raw=FALSE)
    {
        //
        // The PHP request parameter parser adds a backslash (\) to
        // quotes ("). Thus a string with quotes is stored as \" at
        // $_REQUEST. The htmlentities function will convert this to
        // \&quote;. If we serialize such a string and put it into a
        // database we will run into problems when deserializing it
        // because the \ will may be stripped at the database layer
        // and the string will be shorten. In this case, it cannot be
        // unserialized. IMHO this is a bug in php 4!
        //
        // To handle the apostrophe safe, it must be converted to it's
        // HTML entitie. Thus we enable ENT_QUOTES at htmlentities().
        //
        // Because of this the slashes will be stripped.
        global $UNICODE;

        if ($raw) {
            return $this->_request[$attr];
        }
        else {
            if (@is_array($value = $this->_request[$attr])) {
                return $this->_check_array($value,$UNICODE);
            }
            else {
                if ($UNICODE) {
                    return stripslashes($value);
                } else {
                    return htmlentities(stripslashes($value),ENT_QUOTES);
                }
            }
            if ($UNICODE) {
                return stripslashes($this->_request[$attr]);
            } else {
                return htmlentities(stripslashes($this->_request[$attr]),ENT_QUOTES);
            }
        }
    }


    /**
     * _check_array
     *
     * Recursively check the values of nested array parameters.
     *
     * @access private
     * @param string $param
     * @param string $UNICODE
     * @return string htmlentities($param)
     */
    function _check_array ($param,$UNICODE) {
        if (is_array($param)) {
            foreach ($param as $var => $val) {
                $ary[$var] = $this->_check_array($param[$var],$UNICODE);
            }
            return $ary;
        }
        if ($UNICODE) {
            return stripslashes($param);
        }
        else {
            return htmlentities(stripslashes($param),ENT_QUOTES);
        }
    }


    /**
     * getVar
     *
     * Returns the value of the variable.
     *
     * @access public
     * @param string $var Name of the variable
     * @return misc
     */
    function getVar($var)
    {
        return $this->_vars[$var];
    }


    /**
     * Redirect to another action/view
     *
	 * Sends a HTTP Redirect header to the browser. The $params array holds the
     * parameter => value pairs that will be set for the new uri.
     *
     * @access public
     * @param string | array $params and values array
     */
    function redirect ($action,$plist=array())
    {
        global $CURRENT_ACTION;
        $this->setSession('LAST_ACTION',$CURRENT_ACTION);

        $params = session_name().'='.session_id().'&CA='.$action;
        if (is_array($plist)) {
            foreach ($plist as $var => $value) {
                $params .= '&'.$var.'='.$value;
            }
        }
        else {
            $params .= '&'. $plist;
        }
        if ($this->DEBUG) {
            $this->log("REDIRECT ===> $action",2,'SPAF2/App');
            $this->log("------------------------------------------",2,'SPAF2/App');
        }

        if ($redirect_base = $this->getConfig('REDIRECT_BASE')) {
            $location = 'Location: '.$redirect_base.$_SERVER['SCRIPT_NAME']."?$params";
        }
        elseif (isset($_SERVER['HTTPS'])) {
            $PROTO = 'https://';
            $http_host = $_SERVER['HTTP_HOST'];
            if (!preg_match('/^[a-z\d:\.](-*[a-z\d:\.])*$/',$http_host)) {
                $this->log('Exit while invalid hostname for redirect: '.$http_host,0,'SPAF2/App');
                exit;
            }
            $location = 'Location: '.$PROTO.$http_host.$_SERVER['SCRIPT_NAME']."?$params";
        }    
        else {            
            $PROTO = 'http://';
            $http_host = $_SERVER['HTTP_HOST'];
            if (!preg_match('/^[a-z\d:\.](-*[a-z\d:\.])*$/',$http_host)) {
                $this->log('Exit while invalid hostname given for redirect: '.$http_host,0,'SPAF2/App');
                exit;
            }
            $location = 'Location: '.$PROTO.$http_host.$_SERVER['SCRIPT_NAME']."?$params";
        } 

        header($location);
        exit();
    }


    /**
     * Access session parameters.
     *
     * @access public
     * @param string identifier
     * @return misc
     */
    function getSession ($identifier)
    {
        if (array_key_exists($identifier,$this->_session)) {
            return $this->_session[$identifier];
        }
        else {
            return FALSE;
        }
    }


    /**
     * set parameter to session
     * 
     * @access public
     * @param string $identifier
     * @param misc $data
     */
    function setSession ($identifier,$data)
    {
        $this->_session[$identifier] = $data;
    }

                                                                                    /**
     * remove parameter from session
     *                                                                               * @access public
     * @param string $identifier
     */
    function delSession ($identifier)
    {
        unset ($this->_session[$identifier]);
    }
    

    /**
     * getErrorList
     *
     * Get the list of error messages if error is set.
     *
     * @access public
     * @param string $error symbol
     * @return $int error count
     */
    function getErrorList ($error='')
    {
        if ($count = $this->Error->count) {
            return $this->Error->messages;
        }
        else {
            return $count;
        }
    }


    /**
     * getMsg
     *
     * Get the $MSG string of the message symbol.
     *
     * Fetches the message from the file $confdir/ERRORMSG
     *
     * @access public
     * @param string $msg_sym symbol
     * @return string $message
     */
    function getMsg ($msg_sym='')
    {
        $msg_string = '';
        if (! $fd = @fopen ($this->message_file, "r")) {
	        $this->log('ERROR: Cannot read message file '.$this->message_file,0,'SPAF2/App');
        }
        else {
            while (!feof ($fd)) {
                $buffer = fgets($fd, 1024);
                if (preg_match("/^#/",$buffer)) continue;
                if (preg_match("/^\s*$/",$buffer)) continue;
                list($param,$value) = preg_split ("/\s*=\s*/",$buffer,2);
                if ($param == 'ERR_DEFAULT') {
                    $msg_default = rtrim($value);
                    continue;
                }
                if ($param == $msg_sym) {
                    $msg_string = rtrim($value);
                    break;
                }
            }
            fclose($fd);
            if ($msg_string == '') $msg_string = $msg_default;
        }
        return $msg_string;
    }

 
    /**
     * setMsg
     *
     * Set $MSG according to the message symbol.
     *
     * Fetches the message from the file $confdir/ERRORMSG
     *
     * @access public
     * @param string $msg_sym symbol
     * @return $bool TRUE
     */
    function setMsg ($msg_sym='')
    {
        $msg_string = $this->getMsg($msg_sym);
        $this->assign('MSG',$msg_string);
        return TRUE;
    }
 
    
    /**
     * setError
     *
     * Set error message according to the error symbol.
     *
     * Fetches the error message from the file $confdir/error.msg.
     *
     * @access public
     * @param string $error symbol
     * @return $int error count
     */
    function setError ($error)
    {
        if ($error) {
            $count = $this->Error->setError($error);
        } 
        return $count;
    }

    
    /**
     * Sets the application variable
     *
     * @access public
     * @param string $var Name of the parameter
     * @return string $var value of the parameter
     */
    function setVar($var,$value)
    {
        return $this->_vars[$var] = $value;
    }

}


/**
 * @package Application
 */
/**
 * Debug class
 */

class ApplicationDebug extends Application
{
    /**
     * TPL
     *
     * Template engine (Smarty) object.
     *
     * @access public
     */
    var $TPL;

    /**
     * _request
     *
     * Reference to the request parameters.
     *
     * @access public
     */
    var $_request;

    /**
     * _session
     *
     * Reference to the session parameters.
     *
     * @access public
     */
    var $_session;
    

    /**
     * _debug_vars
     *
     * Reference to the debug vars of the application.
     *
     * @access public
     */
    var $_debug_vars;

    /**
     * _output
     *
     * Output string.
     *
     * @access public
     */
    var $_output;

    /**
     * Constructor
     *
     * @param object $Smarty
     * @return object
     */
    public function __construct(&$TPL,&$request,&$session,&$debug_vars)
    {
        $this->TPL = &$TPL;
        $this->_request = &$request;
        $this->_session = &$session;
        $this->_debug_vars = &$debug_vars;
    }

    /**
     * debug
     *
     * Show the debug information depending of the debugging value.
     *
     * 1 = REQUEST Parameters
     * 2 = SESSION Parameters
     * 3 = REQUEST + SESSION
     * 256 = Smarty debugging console
     *
     * @access public
     */
    function debug ($debug_val)
    {
        if ($this->_isBitSet($debug_val,1))    $this->request();
        if ($this->_isBitSet($debug_val,2))    $this->session();
        if ($this->_isBitSet($debug_val,3))    $this->variable();
        if ($this->_isBitSet($debug_val,9))    $this->template();

        // custom debug template
        //$this->compile(SMARTY_DIR.'DEBUG.tpl');
        $this->compile($this->TPL->template_dir.'/DEBUG.tpl');
        return $this->_output;
    }


    /**
     * smarty
     *
     * Show Smarty debugging console.
     *
     */
    function template () {
        $this->TPL->debugging = TRUE;
    }


    /**
     * request
     *
     * Show values of request parameters.
     *
     */
    function request () {
        $vars = array(); $vals = array();
        $request = $this->_request;
        ksort($request);
        foreach ($request as $var => $val) {
            if ($var == 'DEBUG' || $var == session_name())
                continue;
            array_push($vars,$var);
            array_push($vals,$val);
        }
        $this->TPL->assign('DEBUG_REQUEST','1');
        $this->TPL->assign('_request_params',$vars);
        $this->TPL->assign('_request_values',$vals);
     }


    /**
     * session
     *
     * Show values of session parameters.
     *
     */
    function session () {
        $vars = array(); $vals = array();
        $request = $this->_session;
        ksort($request);
        foreach ($request as $var => $val) {
            if ($var == 'DEBUG' || $var == session_name())
                continue;
            array_push($vars,$var);
            array_push($vals,$val);
        }
        $this->TPL->assign('DEBUG_SESSION','1');
        $this->TPL->assign('_session_params',$vars);
        $this->TPL->assign('_session_values',$vals);
    }


    /**
     * debug variable
     *
     * Show values of variable
     *
     */
    function variable () {
        $variable = $this->_debug_vars;
        $vars = array(); $vals = array();

        $a = 1; $o = 1; $v = 1;
        foreach ($variable as $pos => $content) {
            if (is_array($content)) {
                ksort($content);
                array_push($vars,'<b>Array_'.$a++.'</b>');
                array_push($vals,' ');
                foreach ($content as $var => $val) {
                    array_push($vars,'&nbsp;'.$var.' => ');
                    array_push($vals,$val);
                }
            }
            elseif ($class = get_class($content)) {
                array_push($vars,'<b>Object_'.$o++.':</b>');
                array_push($vals,ucfirst($class));
                $ovars = get_object_vars($content);
                foreach ($ovars as $var => $val) {
                    array_push($vars,'&nbsp;->'.$var);
                    array_push($vals,$val);
                }
            }
            else {
                array_push($vars,'Variable: '.$v++);
                array_push($vals,$content);
            }
        }   
        $this->TPL->assign('DEBUG_VARIABLE','1');
        $this->TPL->assign('_variable_params',$vars);
        $this->TPL->assign('_variable_values',$vals);
    }

    // check bitmask
    function _isBitSet($mask,$num)
    {
        return (($mask& (0x01 << ($num-1))));
    }
}

// require_once 'SPAF2/ApplicationError.php';

/**
 * ApplicationError
 */
class ApplicationError extends Application
{
    /**
     * error counter
     *
     * @var int $count
     * @access public
     */
    var $count;

    /**
     * error messages array
     *
     * @var array $messages
     * @access public
     */
    var $messages;

    /**
     * _errormsg
     *
     * @var string $_errormsg
     * @access public
     */
    var $_errormsg;

    
    /**
     * constructor
     *
     * @param string $msgfile message file
     * @access public
     */
    function ApplicationError ($msgfile)
    {
        $this->count = 0;
        $this->_errormsg = $msgfile;
        $this->messages = array();
    }


    /**
     * getLast
     *
     * Returns the last error message set.
     *
     * @access public
     * @return string errormessage
     */
    function getLast ()
    {
        return end($this->messages);
    }

    /**
     * setError
     *
     * Set error message.
     *
     * Fetches the error message from the file $confdir/error.msg.
     *
     * @access public
     * @param string $error symbol
     * @return $int error count
     */
    function setError ($error)
    {
        $file = $this->_errormsg;
        if (file_exists($file)) {
            $fd = fopen ($file, "r");
            $notfound = 1;
            while (!feof ($fd)) {
                $buffer = fgets($fd, 1024);
                if (preg_match("/^$/",$buffer)) {
                    continue;
                }
                if (preg_match("/^#/",$buffer)) {
                    continue;
                }
                @list($sym,$msg) = preg_split("/\s*=\s*/",$buffer);
                if ($sym == $error) {
                    $notfound = '';
                    $this->count = array_push($this->messages,$msg);
                }
            }
            fclose($fd);
        }
        else {
            $notfound = 1;
        }
        if ($notfound == 1) {
            $this->log("Undefined error sym: $error",0,'SPAF2/App');
            $this->count = array_push($this->messages,$error);
        }
        return $this->count;
    }
}

?>
