<?php
/*
 * Benno MailArchiv
 *
 * Copyright  2008-2012 LWsystems GmbH & Co. KG
 *
 * http://www.lw-systems.de/
 * http://www.benno-mailarchiv.de/
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation.
 *
 * Binary versions of this file provided by LWsystems to you as
 * well as other copyrighted, protected or trademarked materials like
 * logos, graphics, fonts, specific documentations and configurations,
 * cryptographic keys etc. are subject to a license agreement between
 * you and LWsystems.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of the GNU Affero General Public License at
 * this URI:
 * http://www.gnu.org/licenses/agpl-3.0.html
 * If not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA  02111-1307  USA
 */

/** $Id: $
 *
 */
/**
 * @package User
 */
/**
 *
 * Abstract class to access User object data from storage
 *
 */
require_once 'SPAF2/Persistence.php';
require_once 'model/Filter.php';


class UserDA extends Persistence
{

    /**
     * FetchList
     *
     * Fetch a list of User objects from database
     *
     * @access public
     * @return array entries
     */
    public static function FetchList ($archive)
    {
        global $App;
        $entries[] = '';
        list($cid,$scid) = explode('/',$archive,2);
        $scid || $scid = '';        // empty values must be set explicit

        $DB = $App->DB;
        $sql = "SELECT userid FROM container WHERE cid=? and scid=? ORDER BY userid";
        $sth = $DB->db->prepare($sql);
        $sth->bindValue(1,$cid);
        $sth->bindValue(2,$scid);
        try {
            if($sth->execute()) {
                while ($row = $sth->fetch()) {
                    $obj = new User($row['userid']);
                    $obj->load();
                    if (!$obj->name) $obj->name = $obj->id;
                    $entries[$obj->id] = $obj;
                }
            }
        } catch (Exception $e) {
            $App->log('Error access container table: '.$e->getMessage());
        }

        if (empty($entries)) { // deprecated data from user table
            $sql = "SELECT * FROM user WHERE id=?";
            $sth = $DB->db->prepare($sql);
            foreach($userids as $uid) {
                $sth->bindValue(1,$cid);
                if ($sth->execute()) {
                    while ($row = $sth->fetch()) {
                        $obj = new User($row['id']);
                        $obj->load();
                        if (!$obj->name) $obj->name = $obj->id;
                        $entries[$obj->id] = $obj;
                    }
                }
            }
        }

        return $entries;
    }

    // Overwrites Persistence::load()
    function load ()
    {
        global $App;
        $DB = $App->DB;

        $sql = 'SELECT * FROM user WHERE id = ?';
        $sth = $DB->db->prepare($sql);
        $sth->bindValue(1,$this->id);
        $res = $sth->execute();
        $row = $sth->fetch();
        $this->password = $row['password'];
        $this->name = $row['name'];
        $this->role = $row['role'];
        $this->load_role_permissions();

        $this->_load_container();

        // read address data
        $sql = 'SELECT address FROM address WHERE id = ?';
        $sth = $DB->db->prepare($sql);
        $sth->bindValue(1,$this->id);
        $res = $sth->execute();
        while($row = $sth->fetch()) {
            array_push($this->addresses,$row['address']);
        }

        return FALSE;
    }


    // load userdata from external userdb via script parameters
    function loadExternal ($script,$loadpass=false)
    {
        global $App;

        if (!is_executable($script)) {
            $App->log('Authentication script "'.$script.'" is not executable.',2);
            throw new Exception('ERR_EXECUTE');
        }

        try {
            return $this->loadSExternal($script,$loadpass);
        }
        catch (Exception $e) {
            $exc = $e->getMessage();
            $App->log("Authentication error: $exc",2,'User/loadExternal');
              if ($App->getConfig('DISABLE_DEPRECATED') === true) {
                  if ($exc == 'ERR_NOUSER') {
                      throw new Exception('ERR_AUTH');    // disguise unknown username
                  }
                  throw new Exception($exc);
              }
        }

        if ($loadpass === false) {
            $password = escapeshellcmd($this->password);
        }
        else {
            $password = escapeshellcmd($loadpass);
            $this->password = $loadpass;
        }

        $cmd = escapeshellcmd($script);
        $username = escapeshellcmd($this->id);
        $App->log('(DEPRECATED) Call "'.$script.'" to authenticate.',0,'User/loadExternal');
        $retvalue = shell_exec("$cmd \"$username\" \"$password\"");
        return $this->_check_external_return($retvalue);
    }


    // load userdata from external userdb via script stdin/out
    function loadSExternal ($script,$loadpass=false)
    {
        global $App;

        if ($loadpass === false) {
            $password = $this->password;
        }
        else {
            $password = ($loadpass);
        }
        $username = $this->id;

        $fd = array(
            0 => array('pipe', 'r'),
            1 => array('pipe', 'w'),
            2 => array('pipe', 'w')
        );

        $cmd = escapeshellcmd($script);
        $process = proc_open($cmd, $fd, $pipes);
        $App->log('Call "'.$script.'" to authenticate.',2);

        $auth_sep = ' ';
        // check for whitespace (FS#140)
        if (strpos($username,' ') !== false) {
            $auth_sep = "\n"; 
            $App->log('WARN: Username contains whitespace: Use \n as separator');
        }
        if (strpos($password,' ') !== false) {
            $auth_sep = "\n"; 
            $App->log('WARN: Password contains whitespace: Use \n as separator');
        }

        if (is_resource($process)) {
            // write "<username>$auth_sep<password>"
            fwrite($pipes[0], $username.$auth_sep.$password);
            fclose($pipes[0]);

            // read stdout from script
            $retvalue = stream_get_contents($pipes[1]);
            fclose($pipes[1]);

            $errvalue = stream_get_contents($pipes[2]);
            fclose($pipes[2]);
            if ($errvalue) {
                $error_lines = '';
                $loglevel = 3;
                foreach (explode("\n", $errvalue) as $error_line) {
                    if (preg_match('/FATAL/',$error_line)) {
                        $loglevel = 0;
                    }
                    $error_lines .= $error_line.'|';
                }
                $error_lines = rtrim($error_lines, '|');
                $App->log('Authentication script error: '.$error_lines,$loglevel,'User/loadSExternal');
            }

            $return_value = proc_close($process);
        }
        // Returnvalues:
        //  ERROR ERR_AUTH | ERR_INTERN | (symbol in benno-errors.txt)
        //  ROLE USER | REVISOR | ADMIN
        //  DISPLAYNAME Martin Werthmöller
        //  ARCHIVE LWsystems
        //  MAIL mw@lw-systems.de
        //  MAIL m.werthmoeller@lw-systems.de
        //  MAIL info@lw-systems.de
        //  USERID <internal_userid>
        return $this->_check_external_return($retvalue);
    }


    // Overwrites Persistence::save()
    function save ($suffix='')
    {
        global $App;

        if ($suffix) {
            // username suffix to separate users in multi domain setup
            $new_id = '';
            if (strstr($this->id,'@')) {
                list ($pfx,$sfx) = explode('@',$this->id,2);
                $new_id = $pfx.'@'.$suffix;
            }
            else {
                $new_id = $this->id.'@'.$suffix;
            }
            $this->id = $new_id;
        }

        $DB = $App->DB;
        $DB->startTransaction();
        try {
            $sql = 'DELETE FROM address WHERE id = ?';
            $sth = $DB->db->prepare($sql);
            $sth->bindValue(1,$this->id);
            $res = $sth->execute();
        }
        catch(PDOException $e){ 
            $App->log('WARN del address: '.$e->getMessage(),2);
        }


        try {
            $sql = "INSERT INTO address (id,address) VALUES (?,?)";
            $sth = $DB->db->prepare($sql);
            foreach ($this->addresses as $address) {
                $sth->bindValue(1,$this->id);
                $sth->bindValue(2,$address);
                $res = $sth->execute();
            }
        }
        catch(PDOException $e){ 
            $App->log('WARN insert address: '.$e->getMessage(),2);
        }

        try {
            $containers = '';
            $sql = 'DELETE FROM container WHERE userid = ?';
            $sth = $DB->db->prepare($sql);
            $sth->bindValue(1,$this->id);
            $res = $sth->execute();
        }
        catch(PDOException $e){ 
            $App->log('WARN del container: '.$e->getMessage(),2);
        }

        try {
            $sql = "INSERT INTO container (userid,cid) VALUES (?,?)";
            $sth = $DB->db->prepare($sql);
            foreach ($this->archive as $container) {
                if ($container == 'SYSDATA') { continue; }
                $sth->bindValue(1,$this->id);
                $sth->bindValue(2,$container);
                $res = $sth->execute();
 
                $containers .= $container.','; // deprecated
            }
            $containers = rtrim($containers,','); // deprecated
        }
        catch(PDOException $e){ 
            $App->log('WARN insert container: '.$e->getMessage(),2);
        }

        try {
            $sql = 'DELETE FROM user WHERE id = ?';
            $sth = $DB->db->prepare($sql);
            $sth->bindValue(1,$this->id);
            $res = $sth->execute();
        }
        catch(PDOException $e){ 
            $App->log('WARN del user: '.$e->getMessage(),2);
        }

        try {
            $sql = "INSERT INTO user (id,name,password,archive,role) VALUES (?,?,?,?,?)";
            $sth = $DB->db->prepare($sql);
            $sth->bindValue(1,$this->id);
            $sth->bindValue(2,$this->name);
            $sth->bindValue(3,$this->password);
            $sth->bindValue(4,$containers);
            $sth->bindValue(5,$this->role);
            $res = $sth->execute();
        }
        catch(PDOException $e){ 
            $App->log('WARN insert user: '.$e->getMessage(),2);
        }
        $DB->endTransaction();
    }


    // Overwrites Persistence::delete()
    function delete ($attributes='')
    {
        global $App;
 
        $DB = $App->DB;
        $DB->startTransaction();
        try {
            $sql = 'DELETE FROM address WHERE id = ?';
            $sth = $DB->db->prepare($sql);
            $sth->bindValue(1,$this->id);
            $res = $sth->execute();

            $sql = 'DELETE FROM appstate WHERE userid = ?';
            $sth = $DB->db->prepare($sql);
            $sth->bindValue(1,$this->id);
            $res = $sth->execute();

            $sql = 'DELETE FROM container WHERE userid = ?';
            $sth = $DB->db->prepare($sql);
            $sth->bindValue(1,$this->id);
            $res = $sth->execute();

            $sql = 'DELETE FROM storedquery WHERE userid = ?';
            $sth = $DB->db->prepare($sql);
            $sth->bindValue(1,$this->id);
            $res = $sth->execute();

            $sql = 'DELETE FROM user WHERE id = ?';
            $sth = $DB->db->prepare($sql);
            $sth->bindValue(1,$this->id);
            $res = $sth->execute();
        }
        catch(PDOException $e){ 
            $App->log('WARN '.$e->getMessage(),0);
        }
        $DB->endTransaction();
    }


    // private
    function _check_external_return($retvalue) {
        global $App;
        $retlines = preg_split('/\n/',$retvalue);
        foreach ($retlines as $line) {
            $line = trim($line);
            if (!$line) continue;
            @list($mark,$data) = preg_split('/\s/',$line,2);
            switch ($mark) {
                case 'ERROR':
                    throw new Exception($data);
                    break;
                case 'WARN':
                    $App->log('WARN '.$data,0);
                    break;
                case 'NOTE':
                    // only for cmdline debugging
                    break;
                case 'ROLE':
                    $this->role = $data;
                    break;
                case 'DISPLAYNAME':
                    $this->name = $data;
                    break;
                case 'ARCHIVE':
                    array_push($this->archive,$data);
                    break;
                case 'MAIL':
                    array_push($this->addresses,$data);
                    break;
                case 'AUTHPARAM':
                    $this->authparam = array_merge(
                            $this->authparam,$this->_parse_param($data));
                    break;
                case 'USERID':
                    $this->id = $data;
                    break;
                case 'FILTER':
                    list($fname,$fstr) = preg_split('/\s/',$data,2);
                    $Filter = new Filter($fname,$this->id,$fname,$fstr);
                    array_push($this->filter,$Filter);
                    break;
                case 'RESTURL':
                    $this->restdata['URL'] = $data;
                    break;
                case 'RESTUSER':
                    $this->restdata['USER'] = $data;
                    break;
                case 'RESTPASS':
                    $this->restdata['PASS'] = $data;
                    break;
            }
        }

        if (!$this->archive) {
            $App->log('[User::external] ARCHIVE not set by external auth',1);
            throw new Exception('ERR_ARCHIVE');
        }
        $this->archive = array_unique($this->archive);

        if (!$this->role)    {
            $App->log('[User::external] ROLE not set by external auth',1);
            throw new Exception('ERR_AUTH');
        }

        if ($this->role == 'SYSTEM') { 
            $App->log('[User::external] SYSTEM ROLE not allowed to login',1);
            throw new Exception('ERR_AUTH');
        }

        $this->load_role_permissions();
        return FALSE;
    }


    // split ':' into array
    function _parse_param($authparam)
    {
        list($param,$value) = preg_split('/\s/',$authparam,2);
        $p[$param] = $value;

        return $p;
    }


    function _load_container()
    {
        global $App;
        $entries = array();

        $DB = $App->DB;
        $sql = "SELECT * FROM container WHERE userid = ? ORDER BY cid,scid";
        try {
            $sth = $DB->db->prepare($sql);
            $sth->bindValue(1,$this->id);
            if($sth->execute()) {
                while ($row = $sth->fetch()) {
                      $cntid = $row['cid'];
                      if ($row['scid'] != '') {
                          $cntid.= '/'.$row['scid'];
                      }
                      array_push($this->archive,$cntid);
                  }
            }
        }
        catch(PDOException $e){ 
            $App->log('WARN '.$e->getMessage(),0);
        }

        // deprecated
        if (empty($this->archive)) {
            $App->log('WARN: container table has no entries for '.$this->id,2);
            $sql = "SELECT archive FROM user WHERE id = ?";
            $sth = $DB->db->prepare($sql);
            $sth->bindValue(1,$this->id);
            if($sth->execute()) {
                $row = $sth->fetch();
                $cntlist = preg_split('/,\s+?/',$row['archive']);
                foreach ($cntlist as $cntid) {
                    if ($cntid) array_push($this->archive,$cntid);
                }
            }
        }
        // /deprecated
    }


    function load_role_permissions()
    {
        global $App;
        $App->log('DEBUG Load role permissions.',4);

        switch($this->role) {
            case 'SITEADMIN':
                $this->setPermission('CONTAINER');
            case 'ADMIN':
                $this->setPermission('ROLE');
                $this->setPermission('ADDRESS');
                $this->setPermission('STATS');
                $this->setPermission('SYSINFO');
            case 'USERADMIN':
                $this->setPermission('USERINFO');
            case 'REVISOR':
                $this->setPermission('SYSDATA');
            case 'USER':
                $this->setPermission('PASSWD');
            case 'AUTH':
                break;    
        }
    }

}
?>
