<?php
/*
 * Benno MailArchiv
 *
 * Copyright  2008-2012 LWsystems GmbH & Co. KG
 *
 * http://www.lw-systems.de/
 * http://www.benno-mailarchiv.de/
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation.
 *
 * Binary versions of this file provided by LWsystems to you as
 * well as other copyrighted, protected or trademarked materials like
 * logos, graphics, fonts, specific documentations and configurations,
 * cryptographic keys etc. are subject to a license agreement between
 * you and LWsystems.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of the GNU Affero General Public License at
 * this URI:
 * http://www.gnu.org/licenses/agpl-3.0.html
 * If not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA  02111-1307  USA
 */

require_once 'model/EMail.php';
require_once 'model/Attachment.php';
require_once 'model/QueryResult.php';
require_once 'model/License.php';

class Backend
{

    var $base_url;
    var $filter;        // filter assign to each query
    var $rest_user;
    var $rest_pass;
    var $_curl;

    function Backend ($base_url,$rest_user,$rest_pass,$filter)
    {
        global $App;
        $this->rest_user = $rest_user;
        $this->rest_pass = $rest_pass;
        $this->rest_ssl_verify = $App->getConfig('REST_SSL_VERIFY','NO');

        $this->base_url = $base_url;
        $this->filter = $filter;
    }


    /**
     * Fetch one email from backend
     *
     * @param string $archive
     * @param string $sessionMailId
     * @return object $EMail
     */
    function fetch ($archive,$sessionMailId)
    {
        global $App;
        $bennoId = QueryResult::getBennoId($sessionMailId);
        $App->log("Fetch mail from archive: $bennoId",3,'Backend');

        $request_url  = $this->base_url.'/mail/';
        $query_string = 'archive='.$archive.
//                        '&filterQuery='.$this->filter.
                        '&id='.$bennoId;

        $maildata = $this->_curl_query($request_url,$query_string);

        $body   = $maildata['body'];
        $header = $maildata['header'];

        $EMail = new EMail($sessionMailId,$archive,$bennoId,$header['From'],$header['To'],$header['Cc'],$header['Bcc'],$header['Subject'],$header['Date'],$header['Size'],$body);
        $EMail->verified = $header['X-Benno-HashCheckPassed'];
        $EMail->dateTS   = $header['X-Benno-DateTS'];

        foreach ($maildata['attachments'] as $atm) {
            //$Attachment = new Attachment($atm['name'],@$atm['content-type'], $atm['size']);
        	$Attachment = new Attachment($atm['name'],@$atm['content-type'], $atm['size'], true, $atm['attachmentId']);
            $EMail->addAttachment($Attachment);
        }

        return $EMail;
    }


    /**
     * Fetch the headers of an email from backend
     *
     * @param string $archive
     * @param string $sessionMailId
     * @return array $headers
     */
    function fetchHeaders ($sessionMailId)
    {
        $bennoId = QueryResult::getBennoId($sessionMailId);

        $request_url  = $this->base_url.'/mail/';
        $query_string = 'archive='.$archive.
//                        '&filterQuery='.$this->filter.
                        '&id='.$bennoId.
                        '&format=headers';

        $header = $this->_curl_query($request_url,$query_string,'headers');

        return $header;
    }


    /**
     * Fetch one email in raw format from backend
     *
     * @param string $archive
     * @param string $sessionMailId
     * @return string $maildata
     */
    function fetchRaw ($archive,$sessionMailId,$recodeToUTF8=false)
    {
        $bennoId = QueryResult::getBennoId($sessionMailId);

        $request_url  = $this->base_url.'/mail/';
        $query_string = 'archive='.$archive.
//                        '&filterQuery='.$this->filter.
                        '&id='.$bennoId.
                        '&format=raw';
        if (!$recodeToUTF8) {
            $query_string .= '&skipUTF8Recode=true';
        }

        $maildata = $this->_curl_query($request_url,$query_string,'raw');
        $result = json_decode($maildata, true);

        return $maildata;
    }


    /**
     * Fetch attachment from backend
     *
     * @param string $archive
     * @param string $mailId
     * @return object $EMail
     */
    function fetchAttachment ($archive,$sessionMailId,$filename, $attachmentId = '')
    {
        $bennoId = QueryResult::getBennoId($sessionMailId);

        $request_url  = $this->base_url.'/attachment/';
        $query_string = 'archive='.$archive.
                        '&id='.$bennoId.
                        '&attachment='.$filename;
                        
	    if($attachmentId != "")         
	      $query_string .= '&attachmentId='.$attachmentId;

        $file = $this->_curl_query($request_url,$query_string,'raw');
        if (! $file) {
            throw new Exception('NO_ATTACHMENT');
        }

        return $file;
    }


    /**
     * Fetch license data from backend
     *
     * @return object $License
     */
    function fetchLicense()
    {
        global $App;

        $request_url  = $this->base_url.'/license/';

        $data = $this->_curl_query($request_url);

        if (! $data) {
            $App->log('[Backend/fetchLicense] No valid License found');
        }

        $License = new License($data['licenseStatus'],$data['licenseExpires'],$data['licenseType'],$data['licenseFile'],$data['myAddress'],$data['myHostname'],$data['archiveCount'],$data['coreVersion'],$data['restVersion']);

        return $License;
    }



    /**
     * Search backend for mails
     *
     * @param object $Query
     * @param start $startnum (default: 1)
     * @param int $limit (default: 20)
     * @return array $EMail objects
     */
    function query ($archive,$Query,$startNum=0,$limit=20)
    {
        global $App;
        global $Audit;

        if (empty($this->filter)) {
            $this->filter = 'NOT*';
        }

        $request_url  = $this->base_url.'/search/';
        $query = 'archive='.$archive .
                 '&filterQuery='.$this->filter .
                 '&limit='.$limit .
                 '&start='.$startNum .
                 '&sort='.$Query->getSortField() .
                 '&sortAsc='.$Query->getSortAsc() .
                 '&query='.$Query->getQuery();

        $App->log("Query string: $query",3,'Backend');
        $App->log("Filter string: $query",4,'Backend');

        $resultList = $this->_curl_query($request_url,$query);

        if (! preg_match('/^LICENSE_VALID/',$resultList['licenseValid'])) {
            if ($App->getConfig('USERPERMISSONS','ENABLED') != 'DISABLED') {
                $App->setMsg('LICENSE_ERROR');
                throw new Exception('LICENSE_ERROR');
            }
        }
        $Audit->log("QUERY $archive:".$Query->getQuery().' && '.$this->filter);

        $QueryResult = new QueryResult($resultList,$Query,$limit);
        return $QueryResult;
    }


    /**
     * Sends http query to benno2rest backend server.
     *
     * @param string $request_url
     * @return array
     */
    function _curl_query ($request_url,$query=false,$format='')
    {
        global $App;
        $cookie_file = $App->getConfig('COOKIE_FILE','/var/tmp/.benno.cookie')
                      .$App->getSession('USER')->id;

        if (!$this->_curl) {
            if (extension_loaded('curl')) {
                $this->_curl = curl_init();
            }
            else {
                $App->log('ERROR: curl module not installed!');
                throw new Exception('ERR_INTERN');
            }
        }
        $curl =& $this->_curl;

        curl_setopt($curl, CURLOPT_URL, $request_url);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_HEADER, 1);
        curl_setopt($curl, CURLOPT_POST, 1);
        curl_setopt($curl, CURLOPT_POSTFIELDS, $query);
        // suppress using of 100-Continue requests
        curl_setopt($curl, CURLOPT_HTTPHEADER,array("Expect:"));
        curl_setopt($curl, CURLOPT_USERPWD, $this->rest_user.':'.$this->rest_pass); 
        if ($cookie_file) {
            curl_setopt($curl, CURLOPT_COOKIEJAR, $cookie_file);
            curl_setopt($curl, CURLOPT_COOKIEFILE, $cookie_file);
        }
        if (preg_match('/^https:/',$request_url)) {
            if ($this->rest_ssl_verify == 'YES') {
                curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, true);
                curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, true);
            }
            else {
                curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, false);
            }
        }

        $App->log('Request URL: '.$request_url."?".$query,4,'Backend');
        if (($response = curl_exec($curl)) === false) {
            $errnum = curl_errno($curl);
            $App->log('[Backend/query] Request error: '.curl_error($curl).'['.$errnum.']');
            if ($errnum == 7) {
                throw new Exception('ERR_REST_CON');
            }
            else {
                throw new Exception('ERR_REQUEST');
            }
        }
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE); 
        if ($httpCode != 200) {
            $App->log('[Backend/query] HTTP error: '.$httpCode);
            throw new Exception('ERR_REST_'.$httpCode);
        }

        // split header / body
        list($response_headers, $response_body) = explode("\r\n\r\n", $response, 2);
        if ($format == 'raw') {
            $result = $response_body;
        }
        else {
            if ( !mb_check_encoding($response_body, 'UTF-8') )
                $response_body = mb_convert_encoding($response_body, 'UTF-8');

            $result = json_decode($response_body, true);
        }

        if (@array_key_exists('error',$result)) {
            $App->log('[Backend/query] '.$result['error']);
            throw new Exception($result['error']);
        }

        return $result;
    }

    function destroy_bennorest_session()
    {
        $request_url = $this->base_url.'/logout/';
        $this->_curl_query($request_url);
    }

}
?>
