package Benno::LDAP::Config;
use Benno::Config;
use parent qw(Benno::Config);

=head1 NAME

    Benno::LDAP::Config

=head1 SYNOPSIS

    use C<Benno::LDAP::Config>;

=head1 DESCRIPTION

    Common used methods and functions.

=head1 CLASS METHODS

=head2 Benno::LDAP::Config->format_addresslist(@addresses)

    Returns list of uniqe email addresses.
    Strips leading "smtp:" from AD/Exchange and encapsulations (<>).

=cut
# format_addresslist
sub format_addresslist
{
    my ($class,@addresslist) = @_;

    my $address;
    my %return_addresses;

    foreach my $token (@addresslist) {
        # strip MS Exchange smtp: prefix
        $token =~ s/^smtp:\s*//i;
        $token =~ s/\s//g;  # FS#1004
        # extract plain address
        #                        (emailprefix|<empty>)@domain
        ($address) = $token =~ /(([a-zA-Z0-9_\*\.+-]+|)\@[a-zA-Z0-9-\*\.]+)/;
        next if $address eq '';

        # delete duplicates
        $return_addresses{$address} = 1;
    }
    # delete duplicates
    return (sort keys %return_addresses);
}


=head2 Benno::LDAP::Config->get_before-created($UserEntry)

    Fetches the "whenCreated" attribute value from LDAP and returns a
    Lucene filter string "NOT SortableDate..." to allow the user only
    access emails after this timestamp.

=cut
sub get_before_created
{
    my ($class,$Conf,$UserEntry) = @_;
    return undef if ($Conf->get('filter_before_created') ne 'true');

    my $filter = 'NOT SortableDate:[197001010000 TO ###ENTRYDATE###]';
    my $created = $UserEntry->get_value('whenCreated');
    my ($entrytime) = $created =~ /^(\d{8})/;
    $filter =~ s/###ENTRYDATE###/${entrytime}000000/;

    return 'StartDate:'.$entrytime.' '.$filter 
}



=head1 METHODS
=head2 new ($configfile)

    Create Benno::Config object

=cut
sub new
{
    my ($class,$configfile) = @_;

    my $self = {
        userattr              => 'uid',
        objectclass           => 'posixAccount',
        userfilter            => '',
        tls                   => 'false',
        email                 => 'mail',
        alias                 => 'emailAlias',
        addemail              => 'bennoEmailAddress',
        role                  => 'bennoRole',
        container             => 'bennoContainer',  # container attribute
        default_container     => 'BennoContainer',
        adminuser             => 'bennoadmin',
        revisoruser           => 'revisor',
        remove_domainsuffix   => 'false',
        user_create_filter    => 'false',
        groupobjectclass      => 'posixGroup',
        groupuserattr         => 'memberUid',
        groupattr_has_dn      => 'false',
        filter_before_created => 'false',
        fdnattribute          => 'mxExchDelegateListBL',
    };
    bless $self, $class;

    $self->SUPER::_init($configfile);

    return $self;
}

### EOP ###
1;

