<?php
/*
 * Benno MailArchiv
 *
 * Copyright  2008-2012 LWsystems GmbH & Co. KG
 *
 * http://www.lw-systems.de/
 * http://www.benno-mailarchiv.de/
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation.
 *
 * Binary versions of this file provided by LWsystems to you as
 * well as other copyrighted, protected or trademarked materials like
 * logos, graphics, fonts, specific documentations and configurations,
 * cryptographic keys etc. are subject to a license agreement between
 * you and LWsystems.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of the GNU Affero General Public License at
 * this URI:
 * http://www.gnu.org/licenses/agpl-3.0.html
 * If not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA  02111-1307  USA
 */

/* $Id: Application.php 27:516c6f1d1ed6 2012-02-19 15:01 +0100 mw $ */
/**
 * @package Application
 */
/**
 * Application base class
 *
 * The applciation class uses special session variables. The following
 * session variables are reserved:
 *
 *   APP_AUTHZLEVEL
 *
 */
@include_once 'lib/Smarty.class.php';

require_once 'SPAF2/ApplicationConfig.php';
require_once 'SPAF2/DBAccess.php';

class Application extends ApplicationConfig
{
    /**
     * authz
     *
     * Array with authorization mappings.
     *
     * @access public
     * @var array
     */
    var $authz;

    /**
     * array with config parameters
     *
     * @access public
     * @var array
     */
    var $config;

    /**
     * _output
     *
     * Holds the output string.
     *
     * @access public
     */
    var $_output;

    /**
     * _request
     *
     * Array with request data.
     *
     * @access public
     * @var array
     */
    var $_request;

    /**
     * module
     *
     * The name of the current module or '.' if main module.
     *
     * @access public
     * @var string
     */
    var $module;


    /**
     * DEBUG
     *
     * @access public
     * @var string
     */
    var $DEBUG;

    /**
     * configfile
     *
     * Main configuration file of this application.
     *
     * @access public
     * @var string
     */
    var $configfile;

    /**
     * configdir
     *
     * The configuration directory of this application is the
     * dirname() of the configuration file.
     *
     * @access public
     * @var string
     */
    var $configdir;


    /**
     * _session
     *
     * Array with session data.
     *
     * @access public
     * @var array
     */
    var $_session;

    /**
     * _vars
     *
     * Array with application variables.
     *
     * @access public
     * @var array
     */
    var $_vars;

    /**
     * _debug_var
     *
     * Holds variables to debug.
     *
     * @var array
     * @access public
     */
    var $_debug_vars;

    /**
     * _TPL
     *
     * Holds the templacte engine class.
     *
     * @access public
     */
    var $TPL;



    /**
     * lang
     *
     * Holds the name of the language setting
     *
     * @access public
     */
    var $lang;


    /**
     * date_format
     *
     * Holds the current date format 
     *
     * @access public
     */
    var $date_format;


    /**
     * time_format
     *
     * Holds the current time format 
     *
     * @access public
     */
    var $time_format;


    /**
     * locale_msg
     *
     * Filename of the message file
     *
     * @access public
     */
    var $locale_msg;


    /**
     * locale_txt
     *
     * Filename of the localized texts
     *
     * @access public
     */
    var $locale_txt;


    /**
     * skin
     *
     * Name of the skin(dir)
     *
     * @access public
     */
    var $skin;

    /**
     * template
     *
     * Holds the current template to render.
     *
     * @access public
     */
    var $template;


    /**
     * tpl_root 
     *
     * Directory with skindirs and message files
     *
     * @access public
     */
    var $tpl_root;


    /**
     * view
     *
     * The current view.
     *
     * @access public
     */
    var $view;


    /**
     * DB
     *
     * If set, holds the DBAccess object of the current application.
     *
     * @access public
     */
    var $DB;


    /**
     * constructor
     *
     * The "configfile" points to current configfile.
     *
     * The Constructor will access the config files 'application.cfg'
     * and 'authz.cfg'.
     *
     * @access public
     * @param string $configfile
     * @param array $_REQUEST request parameters
     * @return object
     */
    function __construct($configfile,$request,$MODULE)
    {

        $this->_debug_vars = array();
        $this->configdir = dirname($configfile);

        $AppConfig = new ApplicationConfig($this->configdir);
        $this->config = $AppConfig->load(basename($configfile));
        $this->authz = $AppConfig->load($this->getConfig('AUTHORIZATION','benno-authz.conf'));

        $this->module   = $MODULE;
        $this->_request = & $_REQUEST;
        $this->_session = & $_SESSION;
        $this->DEBUG = $this->getConfig('DEBUG');

        $this->tpl_root = getcwd().'/'.$this->getConfig('TEMPLATE_ROOT','../')
                          .'/templates';
        $this->TPL = new Smarty;
        $this->TPL->compile_check = true;

        if (isset($this->config['DBTYPE'])) {
            $this->DB = new DBAccess($this->config['DBTYPE'],$this->config['DBUSER'],
                                     $this->config['DBPASS'],$this->config['DBHOST'],
                                     $this->config['DATABASE']);
        }
        else {
            $this->DB = new DBAccess('sqlite:////var/lib/benno-web/bennoweb.sqlite');
        }


    }


    /**
     * Assign the variable to the template
     *
     * @access public
     * @param string $name Name of the template variable
     * @param string $var  Variable to assign
     */
    function assign ($name,$var)
    {
        $this->TPL->assign($name,$var);
    }


    /**
     * authz
     *
     * Check if the current user is allowed to access the given
	 * action. Sets the error ERR_AUTHZ if the current user is not authorized
     * to access the given action.
     *
     * @access public
     * @param string $action
     * @return bool
     */
    function authz ($action)
    {
        $authorized = FALSE;
        if (isset($this->authz[$action])) {
            if ($this->accessLevel() >= $this->authz[$action]) {
                $authorized = TRUE;
            }
        }
        if (! $authorized) {
            $this->setMsg('ERR_AUTHZ');
        }
        return $authorized;
    }


    /**
     * setAuthz
     *
     * Set authorization level for the given action in current response.
     *
     * @access public
     * @param string $action
     * @param string $level
     * @return previous level
     */
    function setAuthz($action,$level)
    {
        $prev_level = $this->authz[$action];
        $this->authz[$action] = $level;

        return $prev_level;
    }


    /**
     * Return the value of a configuration parameter.
     *
     * @access public
     * @param string $parameter
     * @param string $default (optional)
     * @return misc $value
     */
    function getConfig($parameter,$default='')
    {
        if (isset($this->config[$parameter])) {
            $parameter = $this->config[$parameter];
        }
        else {
            $parameter = $default;
        }
        if (preg_match('/^(true|on|yes)$/i',$parameter)) {
            $parameter = true;
        }
        if (preg_match('/^(false|off|no)$/i',$parameter)) {
            $parameter = false;
        }

        return $parameter;
    }


    /**
     * Get array of config values
     *
     * @access public
     * @param string $name
     * @param string $default (optional)
     * @return arrray $params
     */
    function getConfigList ($name,$default=array())
    {
        if (isset($this->config[$name])) {
            return  preg_split('/,(\s+)?/',$this->config[$name]);
        }
        else if ($default) {
            return $default;
        }
        else {
            return $default;
        }
    }


    /**
     * accessLevel
     *
	 * Returns the authorization level of the current user / session.
     *
     * @access public
     * @return int
     */
    function accessLevel ()
    {
        $level = 0;
        if (array_key_exists('ACCESS_LEVEL',$this->_session)) {
            $level = $this->_session['ACCESS_LEVEL'];
        }
        if (! $level) { $level = 0; }
        return $level;
    }


    /**
     * setAccessLevel
     *
	 * Sets the authorization level of the current user / session. Returns the
     * access level.
     *
     * @access public
     * @param int accesslevel
     * @return int
     */
    function setAccessLevel ($level=0)
    {
        $this->_session['ACCESS_LEVEL'] = $level;
        return $this->_session['ACCESS_LEVEL'];
    }


    /**
     * Compile the template and append to output
     *
     * The suffix of the template file (default: .tpl) must not be
     * given. If no suffix given, '.tpl' will appended.
     *
     * @access public
     * @param string $template Template filename
     */
    function compile ($template)
    {
        /* The smarty debuggig could only be used with the display
         * method.
         */
        ob_start();
        //if (file_exists(SMARTY_DIR.'/'.$template)) {
        if (file_exists($template)) {
            $this->TPL->display($template);
            $this->_output .= ob_get_contents();
        }
        elseif(@$this->config['DEBUG'] > 0) {
            $this->log("NO TEMPLATE $template",0,'SPAF2/App');
        }
        ob_end_clean();
    }


    /**
     * configure template set
     *
     * Settings for all templates
     *
     * @access public
     */
    function tplConfig()
    {
        // initialize cache
        $templates_cache_base = $this->getConfig('TEMPLATE_CACHE','/var/cache/benno-web/templates_c');
        $templates_cache_skin = $templates_cache_base.'/'.$this->skin;
        if (!is_dir($templates_cache_skin)) {
            mkdir($templates_cache_skin);
        }
        if (is_writable($templates_cache_skin)) {
            $this->TPL->compile_dir = $templates_cache_skin;
        }
        else {
            $this->log('warning: cannot write to templates cache: '.$templates_cache_skin,0,'spaf2/app');
            $this->TPL->compile_dir = '/tmp';
        }
        $this->TPL->config_dir   = $this->configdir;
        $this->TPL->cache_dir    = $this->tpl_root . '/cache';

        if ($debug = $this->getparam('DEBUG')) {
            $this->setSession('DEBUG',$debug);
        }
    }


    /**
     * Calls (require) the file
     *
     * This method inculdes (require(s)) the file an sets the current
     * Application object to the local variable $App.
     *
     * @access public
     * @param string $file
     * @return misc $return return value of the included file
     */
    function call ($file)
    {
        $App =& $this;          // need by include
        return require $file;
    }


    /**
     * Log something
     *
     * @access public
     * @param string $message
     */
    function log ($message,$threshold=0,$module='')
    {
        $view = '-';
        if ($this->view != '') {
            $view = $this->view;
        }
        if ($module) {
            $view = $module;
        }

        $debug_vals   = preg_split('/,(\s+)?/',$this->DEBUG);
        $debug_global = array_shift($debug_vals);
        foreach ($debug_vals as $debug_entry) {
            list($debug_module,$level) = explode(':',$debug_entry);
            if (($debug_module == $module) && ($level >= $threshold)) {
                error_log("[$threshold:$view] $message");
                return;
            }
        }
        if ($debug_global >= $threshold) {
            error_log("[$threshold:$view] $message");
        }
    }


    /**
     * Returns the loglevel for the given module
     *
     * @access public
     * @param string $message
     * @return int $loglevel
     */
    function loglevelForModule($module)
    {
        $debug_vals   = preg_split('/,(\s+)?/',$this->DEBUG);
        $debug_global = array_shift($debug_vals);

        foreach ($debug_vals as $debug_entry) {
            list($debug_module,$level) = explode(':',$debug_entry);
            if ($debug_module == $module) {
                return $level;
            }
        }
        return $debug_global;
    }


    /**
     * debug_var
     *
     * Perform debugging for one variable
     *
     * Shows the debugging console with the variable.

     * @access public
     * @param misc $variable
     */
    function debug_var ($var)
    {
        array_push($this->_debug_vars,$var);
    }


    /**
     * Initialize template engine and return output
     *
     * @access public
     *
     */
    function display ()
    {
        echo $this->_output;
    }


    /**
     * getParam
     *
     * Returns the attribute string with html entities replaced.
     *
     * This method uses the stripslashes() and htmlentities() functions of PHP
     * to convert the parameter string. 
     *
     * If the parameter $raw has been set, no conversation will be performed on
     * the returned parameter.
     * 
     * If the global variable $UNICODE has been set, the htmlentities function
     * will not be performed on the parameters.
     *
     * @access public
     * @param name
     * @param bool $raw to fetch raw data
     * @return string
     */
    function getParam($attr,$raw=FALSE)
    {
        //
        // The PHP request parameter parser adds a backslash (\) to
        // quotes ("). Thus a string with quotes is stored as \" at
        // $_REQUEST. The htmlentities function will convert this to
        // \&quote;. If we serialize such a string and put it into a
        // database we will run into problems when deserializing it
        // because the \ will may be stripped at the database layer
        // and the string will be shorten. In this case, it cannot be
        // unserialized. IMHO this is a bug in php 4!
        //
        // To handle the apostrophe safe, it must be converted to it's
        // HTML entitie. Thus we enable ENT_QUOTES at htmlentities().
        //
        // Because of this the slashes will be stripped.
        global $UNICODE;

        if ($raw) {
            return $this->_request[$attr];
        }
        else {
            if (@is_array($value = $this->_request[$attr])) {
                return $this->_check_array($value,$UNICODE);
            }
            else {
                if ($UNICODE) {
                    return stripslashes($value);
                } else {
                    return htmlentities(stripslashes($value),ENT_QUOTES);
                }
            }
            if ($UNICODE) {
                return stripslashes($this->_request[$attr]);
            } else {
                return htmlentities(stripslashes($this->_request[$attr]),ENT_QUOTES);
            }
        }
    }


    /**
     * _check_array
     *
     * Recursively check the values of nested array parameters.
     *
     * @access private
     * @param string $param
     * @param string $UNICODE
     * @return string htmlentities($param)
     */
    function _check_array ($param,$UNICODE) {
        if (is_array($param)) {
            foreach ($param as $var => $val) {
                $ary[$var] = $this->_check_array($param[$var],$UNICODE);
            }
            return $ary;
        }
        if ($UNICODE) {
            return stripslashes($param);
        }
        else {
            return htmlentities(stripslashes($param),ENT_QUOTES);
        }
    }


    /**
     * Redirect to another action/view
     *
	 * Sends a HTTP Redirect header to the browser. The $params array holds the
     * parameter => value pairs that will be set for the new uri.
     *
     * @access public
     * @param string | array $params and values array
     */
   function redirect ($action,$plist=array())
    {

        // redirect to another script if "?" given
        @list($script,$ca) = explode('?',$action);
        if ($ca) {
            $redir_to = dirname($_SERVER['SCRIPT_NAME']).'/'.$script;
            $params .= $ca;
        }
        else {
            $redir_to  = $_SERVER['SCRIPT_NAME'];
            $params .= 'CA='.$action;
        }

        if ($this->DEBUG) {
            $this->log("REDIRECT ===> $redir_to?$params",2,'SPAF2/App');
            $this->log("------------------------------------------",2,'SPAF2/App');
        }

        // show sessionid in url
        if ($this->getConfig('URL_SESSIONID') == 'ENABLED') {
            $sidprefix = session_name().'='.session_id().'&';
        }
        $params .= $sidprefix;


        if (is_array($plist)) {
            foreach ($plist as $var => $value) {
                $params .= '&'.$var.'='.$value;
            }
        }
        else {
            $params .= '&'. $plist;
        }

        // HTTP HOST
        if (isset($_SERVER['HTTP_X_FORWARDED_HOST'])) {
            $http_host = $_SERVER['HTTP_X_FORWARDED_HOST'];
        }
        else {
            $http_host = $_SERVER['HTTP_HOST'];
        }
        // HTTP PROTOCOL
        if (isset($_SERVER['HTTP_X_FORWARDED_PROTO'])) {
            $http_proto = $_SERVER['HTTP_X_FORWARDED_PROTO'];
            $http_proto = $http_proto.'://';
        }
        elseif (isset($_SERVER['HTTPS'])) {
            $http_proto = 'https://';
        }    
        else {            
            $http_proto = 'http://';
        }
        // HTTP PORT
        if (isset($_SERVER['HTTP_X_FORWARDED_PORT'])) {
            $http_port = $_SERVER['HTTP_X_FORWARDED_PORT'];
        }
        else {
            $http_port = '';
        }

        // HTTP CONFIGURATION
        if ($redirect_base = $this->getConfig('REDIRECT_BASE')) {
            $location = 'Location: '.$redirect_base.$redir_to."?$params";
        }
        else {
            if ($http_port) {
                $http_host = $http_host.':'.$http_port;
            }
            $location = 'Location: '.$http_proto.$http_host.$redir_to."?$params";
        }

        header($location);
        exit();
    }


    /**
     * Access session parameters.
     *
     * @access public
     * @param string identifier
     * @return misc
     */
    function getSession ($identifier)
    {
        if (is_array($this->_session) && array_key_exists($identifier,$this->_session)) {
            return $this->_session[$identifier];
        }
        else {
            return FALSE;
        }
    }


    /**
     * set parameter to session
     * 
     * @access public
     * @param string $identifier
     * @param misc $data
     */
    function setSession ($identifier,$data)
    {
        $this->_session[$identifier] = $data;
    }

                                                                                    /**
     * remove parameter from session
     *                                                                               * @access public
     * @param string $identifier
     */
    function unsetSession ($identifier)
    {
        unset ($this->_session[$identifier]);
    }
    

    /**
     * getMsg
     *
     * Get the $MSG string of the message symbol.
     *
     * Fetches the message from the file $confdir/ERRORMSG
     *
     * @access public
     * @param string $msg_sym symbol
     * @return string $message
     */
    function getMsg ($msg_sym='')
    {
        $msg_string = '';
        if (! $fd = @fopen ($this->locale_msg, "r")) {
	        $this->log('ERROR: Cannot read message file '.$this->locale_msg,0,'SPAF2/App');
        }
        else {
            while (!feof ($fd)) {
                $buffer = fgets($fd, 1024);
                if (preg_match("/^#/",$buffer)) continue;
                if (preg_match("/^\s*$/",$buffer)) continue;
                list($param,$value) = preg_split ("/\s*=\s*/",$buffer,2);
                if ($param == 'ERR_DEFAULT') {
                    $msg_default = rtrim($value);
                    continue;
                }
                if ($param == $msg_sym) {
                    $msg_string = rtrim($value);
                    break;
                }
            }
            fclose($fd);
            if ($msg_string == '') $msg_string = $msg_default;
        }
        return $msg_string;
    }

 
    /**
     * setLang
     *
     * Sets attributes:
     *
     * ->lang
     * ->locale_msg
     * ->locale_txt
     * ->date_format
     * ->time_format
     *
     * @access public
     * @param string $locale
     */
    function setLang($language)
    {
        $default_language = $this->getConfig('DEFAULT_LANGUAGE','de');
        if ($language == 'DEFAULT_LANGUAGE') {
            $language = $default_language;
        }
        $this->lang = $language; 

        // skin localization
        $locale_txt = $this->tpl_root.'/'.$this->lang.'.txt';
        if (!is_readable($locale_txt)) {
            $locale_txt = $this->tpl_root.'/'.$default_language.'.txt';
        }
        $this->locale_txt = $locale_txt;
        // messages
        $locale_msg = $this->tpl_root.'/'.$this->lang.'.msg';
        if (!is_readable($locale_msg)) {
            $locale_msg = $this->tpl_root.'/'.$default_language.'.msg';
        }
        $this->locale_msg = $locale_msg;

        $this->date_format = $this->getMsg('DATE_FORMAT');
        if (!$this->date_format) {
            $this->date_format = '%d.%m.%Y';
        }
        $this->time_format = '%H:%M';
    }


    /**
     * setMsg
     *
     * Set $MSG according to the message symbol.
     *
     * Fetches the message from the file $confdir/ERRORMSG
     *
     * @access public
     * @param string $msg_sym symbol
     * @return $bool TRUE
     */
    function setMsg ($msg_sym='')
    {
        $msg_string = $this->getMsg($msg_sym);
        $this->assign('MSG',$msg_string);
        return TRUE;
    }


    /**
     * clearMsg
     *
     * Set $MSG according to the message symbol.
     *
     * Fetches the message from the file $confdir/ERRORMSG
     *
     * @access public
     * @param string $msg_sym symbol
     */
    function clearMsg ()
    {
        $this->TPL->clearAssign('MSG');
    }


    /**
     * setSkin
     *
     * Set skin directory.
     *
     * @access public
     * @param $skin string
     */
    function setSkin($skindir='',$header_template='header.tpl')
    {
        if (!$skindir) {
            $skindir = 'default';
        }
        $this->skin = $skindir;
        $tpl_dir  = $this->tpl_root.'/'.$skindir;
        if (!is_file($tpl_dir.'/'.$header_template)) {
            $this->log("Warning: Skin $tpl_dir not initialized.",2,'spaf2/app');
            $tpl_dir = $this->tpl_root.'/default';
        }
        $this->TPL->setTemplateDir($tpl_dir);
    }


    /**
     * response
     *
     * Print (raw) response without template and exit.
     *
     * @access public
     * @param $response string
     * @param $type string (default text/plain)
     */
    function response($response,$type='text/plain')
    {
        header("Content-Type: $type");
        print $response;
        $this->log('>>> eor ----------------------------------',1,'SPAF2');
        exit();
    }
}

?>
