<?php
/*
 * Benno MailArchiv
 *

 *
 * http://www.lw-systems.de/
 * http://www.benno-mailarchiv.de/
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation.
 *
 * Binary versions of this file provided by LWsystems to you as
 * well as other copyrighted, protected or trademarked materials like
 * logos, graphics, fonts, specific documentations and configurations,
 * cryptographic keys etc. are subject to a license agreement between
 * you and LWsystems.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of the GNU Affero General Public License at
 * this URI:
 * http://www.gnu.org/licenses/agpl-3.0.html
 * If not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA  02111-1307  USA
 */

/**
 $Id: DBAccess.php 27:516c6f1d1ed6 2012-02-19 15:01 +0100 mw $
*/

/**
 * @package Model
 * @subpackage Persistence
 */
class DBAccess
{
    /**
     *
     * DBAccess
     *
     * Database Handling
     *
     * Configuration will be done in "config.inc.php" where the variables
     *
     * $dbUser
     * $dbPassword
     * $host
     * $database
     *
     * must be set to connect to the database. After connect, the database
     * connection object will be stored at the global var $DB.
     *
     */

    /**
     * Fehlerkennzeichnung
     *
     * Enthaelt einen wahren Wert, wenn die Datenbankabfrage einen
     * Fehler zurueckgibt.
     *
     * @access public
     */
    var $error;


    /**
     * Datenbankobjekt
     *
     * Speichert das DBO Objekt.
     *
     * @access public
     */
    var $db;


    /**
     * Statement handle
     *
     * @access private
     */
    var $_sth;


    /**
     * Constructor
     *
	 * If the sqlite db is used, the $dbType parameter contains the sqlite
     * connect string:
     *
     *    sqlite:///path/to/sqlite/file
     *
     * @access public
     * @param string $dbType 
     * @param string $dbUser (optional)
     * @param string $dbPassword (optional)
     * @param string $host (optional)
     * @param string $database (optional)
     */
    function __construct($dbType,$dbUser='',$dbPassword='',$host='',$database='')
    {

        $this->error = FALSE;

        $this->db_transaction = TRUE;
        if (strpos($dbType,'sqlite') === FALSE) {
            // if not sqlite
             $dsn = "$dbType:host=$host;dbname=$database";
            try {
                $this->db = new PDO($dsn,$dbUser,$dbPassword);
            }
            catch(PDOException $e)
            {
                error_log("[DBAccess] Cannot connect database($dsn): ".$e->getMessage());
            }
        }
        else {
            // sqlite database
            $dsn = $this->_prepare_sqlite_dsn($dbType);
            try {
                $this->db = new PDO($dsn);
            }
            catch(PDOException $e)
            {
                error_log("[DBAccess] Cannot connect database($dsn): ".$e->getMessage());
            }
        }
        $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    }


    /**
     * SQL query
     *
     * Wrapper for SQL queries. If error, the method will log this to
     * error log and triggers a rollback. The query will alway done
     * with FetchMode "DB_FETCHMODE_ASSOC", thus we will fetch an
     * associative array from the DB_result object.
     *
     * @access public
     * @param string SQL query string
     * @param bool LOG_ERROR (default=true)
     * @return object PEAR DB_result object
     */
    function query ($sql,$LOG_ERROR=true)
    {
        try {
            $this->_sth = $this->db->prepare($sql);
            $this->_sth->execute();
        }
        catch(PDOException $e){ 
            if ($LOG_ERROR)
                error_log(' [DBMS/query]: '.$e->getMessage());
        }
        try {
            $result = $this->_sth->fetchAll(PDO::FETCH_ASSOC);
        }
        catch (PDOException $e) {
            $result = '';
        }
        return new DBResult($result);
    }

    
    /**
     * fetchRow
     *
     * Fetch result row from a query directly.
     * Returns the row or an empty value.
     *
     * @access public
     * @param $sql string
     * @return DB_row object
     */
    function fetchRow ($sql)
    {
        if (preg_match('/^(INSERT|UPDATE|DELETE)/i' ,$sql)) {
            return FALSE;
        }
        $this->_sth = $this->db->query($sql);
        $row = $this->_sth->fetch(PDO::FETCH_ASSOC);
		return $row;
    }


    /**
     * start transaction
     *
     * @access public
     */
    function startTransaction ()
    {
        if ($this->db_transaction) {
            $this->error = FALSE;
            $this->db->beginTransaction();
        }
    }


    /**
     * ends transaction
     *
	 * If the transaction was succesful, the method returns TRUE, otherwise
     * FALSE.
     *
     * @access public
     * @return TRUE | FALSE
     */
    function endTransaction ()
    {
        $retval = TRUE;
        if ($this->db_transaction) {
            if ($this->error) {
                $this->db->rollback();
                $retval = FALSE;
            }
            else {
                $this->db->commit();
            }
            $this->error = FALSE;
        }
        return $retval;
    }

    /**
     * Transaktion zuruecksetzen
     *
     * Fuehrt ein 'ROLLBACK' aus, 'db_transaction' auf einen wahren Wert
     * gesetzt ist.
     *
     * @access public
     */
    function rollback ()
    {
        if ($this->db_transaction) {
            $this->db->rollback();
        }
    }

    /**
     * Quote smart
     *
     * @access public
     * @param mixed
     * @return mixed
     */
    function quote ($str)
    {
        return $this->db->quote($str);
    }


    /**
     * Returns the number of affected rows
     *
     * Need for INSERT, UPDATE or DELETE queries.
     *
     * Not supported by all database backends!
     *
     * @access public
     * @return mixed
     */
    function affectedRows ()
    {
        return $this->_sth->rowCount();
    }


    /**
     * check sqlite db file format and return connect string
     *
     */
    function _prepare_sqlite_dsn($dsn)
    {
        $filepath = preg_replace('/^\S+?:/','',$dsn);
        $dbformats = array('sqlite','sqlite2','sqlite3');
        foreach ($dbformats as $dbformat) {
            try {
                $dbh = new PDO($dbformat.':'.$filepath);
                $row = $dbh->query("SELECT * FROM sqlite_master WHERE type='table'");
                $err = $dbh->errorCode();
                if ($err == '0000') {
                    return "$dbformat:$filepath";
                }
            }
            catch(PDOException $e) {
                error_log("[DBAccess/check_sqlite]: ".$dbformat." -> ".$e->getMessage());
            }
        }
    }

}


class DBResult
{
    /**
     *
     * DBResult
     *
     * Compatiblity class implements the PEAR::DB_Result interface partially.
     *
     */
    var $query_result;

    function DBResult($query_result)
    {
        $this->query_result = $query_result;
    }


    function fetchRow()
    {
        $current_row = current($this->query_result);
        next($this->query_result);
        return $current_row;
    }

    function affectedRows()
    {
        return $this->_sth->rowCount();
    }
}

?>
